<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  Authentication.asuser
 *
 * @copyright   (C) 2006 Open Source Matters, Inc. <https://www.joomla.org>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\Authentication\AsUser\Extension;

use Joomla\CMS\Authentication\Authentication;
use Joomla\CMS\Event\User\AuthenticationEvent;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\User\UserFactoryAwareTrait;
use Joomla\CMS\User\UserHelper;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Event\SubscriberInterface;
use Joomla\CMS\Factory;
use Joomla\Event\DispatcherInterface;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Plugin\Authentication\Joomla\Extension\Joomla;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * AsUser Authentication plugin
 *
 * @since  1.5
 */
final class AsUser extends CMSPlugin implements SubscriberInterface
{

    use DatabaseAwareTrait;
    use UserFactoryAwareTrait;

    private $jauth;

    public function __construct(DispatcherInterface $dispatcher, $config = [])
    {
        parent::__construct($dispatcher, $config);
        $jauth = PluginHelper::getPlugin('authentication', 'joomla');
        $this->jauth = new Joomla($dispatcher, (array)$jauth);
    }

    /**
     * Returns an array of events this subscriber will listen to.
     *
     * @return  array
     *
     * @since   5.0.0
     */
    public static function getSubscribedEvents(): array
    {
        return ['onUserAuthenticate' => 'onUserAuthenticate'];
    }

    /**
     * This method should handle any authentication and report back to the subject
     *
     * @param   AuthenticationEvent  $event    Authentication event
     *
     * @return  void
     *
     * @since   1.5
     */
    public function onUserAuthenticate(AuthenticationEvent &$event): void
    {
        $credentials = $event->getCredentials();
        $response    = $event->getAuthenticationResponse();
        $options = $event->getOptions();

        $response->type = 'AsUser';

		// Joomla does not like blank passwords
		if (empty($credentials['password']))
		{
            // Invalid password
            $response->status        = Authentication::STATUS_FAILURE;
            $response->error_message = $this->getApplication()->getLanguage()->_('JGLOBAL_AUTH_INVALID_PASS');

			return;
		}

		$authenticators = $this->_groupUsers($this->params->get('validgroups',array(),'ARRAY'));
		$protected = $this->_groupUsers($this->params->get('invalidgroups',array(),'ARRAY'));
        $protected[] = 8; // Super Users
		$authid = $this->_authUserId($credentials['username']);
        
        if($authenticators && $authid && !$this->_badTarget($authenticators,$authid) && !array_key_exists($authid,$protected)) {
            $match = false;
            foreach($authenticators as $id=>$user) {
                $match = UserHelper::verifyPassword($credentials['password'], $user->password, $id);
                if($match === true){
                    $user = Factory::getUser($authid);
                    $response->email = $user->email;
                    $response->fullname = $user->name;
                    if(Factory::getApplication()->isClient('administrator')) {
                        $response->language = $user->getParam('admin_language');
                    } else {
                        $response->language = $user->getParam('language');
                    }
                    $response->status = Authentication::STATUS_SUCCESS;
                    $response->error_message = '';
                    break;
                }
            }
        } else {
            $response->status        = Authentication::STATUS_FAILURE;
            $response->error_message = $this->getApplication()->getLanguage()->_('JGLOBAL_AUTH_NO_USER');
        }
    }
	private function _authUserId($username) {
        $db = $this->getDatabase();
		$query = $db->createQuery(true);
		$query->select('id')->from('#__users')->where('username = '.$db->q(trim($username)));
		$db->setQuery($query);
		$authid = $db->loadResult();
		return $authid;
	}
	private function _groupUsers($groups) {
		if(!count($groups)) return array();
        $db = $this->getDatabase();
		$query = $db->createQuery(true);
		$query->select('u.id AS id,u.password AS password')
			->from('#__user_usergroup_map AS m')
			->join('left','#__users AS u ON u.id = m.user_id')
			->where('m.group_id IN ('.implode(',',$groups).')');
		$db->setQuery($query);
		$result = $db->loadObjectList('id'); // list of users who are allowed to authenticate
		return $result;
	}
	private function _badTarget($authenticators,$userid) {
		$groups = array(8);
		$superadmins = $this->_groupUsers($groups);
		$targets = array_merge($authenticators,$superadmins);
		if(array_key_exists($userid,$targets)) {
			return true;
		}
		return false;
	}
}
