var HashCashClass = function(options) {
    var root = this;
    this.vars = {
        container: null,
        remote_addr: null,
        level: 3,
        request_time: null,
        iterations: null,
        delaystart: true,
        delaymonitor: false,
        listener: null,
        punish: false
    };
    
    Object.assign(root.vars, options);

    var construct = function() {
        const container = document.getElementById(root.vars.container);
        if(!root.vars.delaystart){
            root.startCalculation();
        } else {
            root.vars.listener = container.form;
            root.vars.listener.addEventListener('change',function(e){
                root.startCalculation(e);
            });
        }
    }

    this.startCalculation = async function(e) {
        /* prevent externally triggered events, event MUST be triggered by the listener element */
        if(
            Object.getPrototypeOf(e) !== Event.prototype || /* event is NOT an instance of Event */
            e.currentTarget !== root.vars.listener || /* listener element is not the target */
            e.bubbles === false || /* event is not bubbling - a form element must bubble the event to the listener */
            e.isTrusted === false /* event is not trusted (user initiated) */
        ){
            if(root.vars.trigger) {
                // site operator has opted to trigger an event when a bot is detected
                var detail = {
                    "ip": root.vars.remote_addr,
                    "time": root.vars.request_time,
                    "ua": navigator.userAgent,
                    "event": Object.getPrototypeOf(e) == Event.prototype,
                    "target": e.currentTarget.id,
                    "bubbles":e.bubbles,
                    "trusted":e.isTrusted
                };
                window.dispatchEvent(new CustomEvent('plg_captcha_hashcash', {detail:detail}));
            }
            /* target element is not the listener and/or is not trusted (user initiated), so we do something else */
            if(root.vars.punish){
                // site operator has opted to inflict punishment on the bot attempting to trick the form
                root.vars.level = 32;
            } else {
                // site operator is a saint
                return;
            }
        }
        root.vars.iterations = Math.pow(100, root.vars.level);
        const container = document.getElementById(root.vars.container);
        try {
            var getCash = await root.getCash();
            const result = await getCash();
            container.value = result;
        } catch (error) {
            console.error('Error generating hash:', error);
        }
    }

    this.getCash = async function() {
        const text = root.vars.remote_addr + root.vars.request_time;
        let count = 0;

        return async function() {
            const pattern = new RegExp('^0{' + root.vars.level + '}');
            
            while (root.vars.iterations > 0) {
                try {
                    const hashBuffer = await root.sha256Subtle(text + count);
                    const hashString = Array.from(new Uint8Array(hashBuffer))
                        .map(b => b.toString(16).padStart(2, '0'))
                        .join('');
                    
                    if(pattern.test(hashString)){
                        return count;
                    }
                    count++;
                    root.vars.iterations--;
                } catch (error) {
                    console.error('Error generating hash:', error);
                    count++;
                    root.vars.iterations--;
                }
            }
            return false;
        };
    }

    this.sha256Subtle = async function(text) {
        try {
            const encoder = new TextEncoder();
            const textBuffer = encoder.encode(text);
            const hashBuffer = await crypto.subtle.digest('SHA-256', textBuffer);
            return hashBuffer;
        } catch (error) {
            throw new Error('Failed to generate SHA256 hash: ' + error.message);
        }
    }

    construct();
};

// DOM load event listener
window.addEventListener('DOMContentLoaded', ()=>{
    new HashCashClass(Joomla.getOptions('plg_captcha_hashcash'));
});