let HashCashClass = function(options) {
    const root = this;
    this.container = null;
    this.worker = null;
    this.vars = {
        success: false,
        remote_addr: null,
        level: 3,
        request_time: null,
        iterations: null,
        delaystart: true,
        delaymonitor: false,
        listener: null,
        punish: false,
        cdp: false,
        trigger: false,
        hasUserInteraction: false,
        isCDPTrapped: false,
        nonce: false,
        nonceValue: null,
        noncefield: null,
        isMining: false
    };    

    const construct = function(options) {
        if (!options || !options.container) {
            console.error('Invalid options provided to HashCashClass');
            return;
        }
        Object.assign(root.vars, options);
        root.container = document.getElementById(options.container);
        if (!root.container) {
            console.error('HashCash container not found:', options.container);
            return;
        }
        ['mousemove', 'keydown', 'click'].forEach(event => {
            document.addEventListener(event, (e) => { 
                if (e.isTrusted) {
                    root.vars.hasUserInteraction = true;
                }
            }, { 
                once: true
            });
        });
        if(root.vars.cdp){
            root.vars.isCDPTrapped = root.testWindowForCDPRuntime(window);
        }
        if (root.vars.nonce) { // Generate and add nonce if enabled
            root.vars.nonceValue = Math.random().toString(36).slice(2);
            root.vars.noncefield = document.createElement('input');
            root.vars.noncefield.type = 'hidden';
            root.vars.noncefield.name = 'hashcash_nonce';
            root.vars.noncefield.value = root.vars.nonceValue;
            root.container.form.appendChild(root.vars.noncefield);
        }
        if(!root.vars.delaystart){
            root.startCalculation();
        } else {
            root.vars.listener = root.container.form;
            root.vars.listener.addEventListener('change', root.handleChange);
        }
    }

    this.startCalculation = async function(event = null) {
        if (root.vars.success || root.vars.isMining) return;
        window.dispatchEvent(new Event('plg_captcha_hashcash_started'));
        const checks = [
            /* event is NOT an instance of Event (therefor artificial) */
            (event) => Object.getPrototypeOf(event) !== Event.prototype,
            /* nonce check if enabled, otherwise normal event element value change test */
            (event) => root.vars.nonce ? (
                event.srcElement.value === event.srcElement.defaultValue && 
                event.srcElement.value !== root.vars.nonceValue
            /* event is not a real change event because nothing changed */
            ) : event.srcElement.value === event.srcElement.defaultValue,
            /* bot manipulated the nonce input */
            (event) => root.vars.nonce && event.srcElement === root.vars.noncefield,
            /* listener element is not the target (only the listener is allowed to send change) */
            (event) => event.currentTarget !== root.vars.listener,
            /* event is not bubbling (listener gets change through bubble) */
            (event) => event.bubbles === false,
            /* event is not trusted (is script initiated) */
            (event) => event.isTrusted === false,
            /* form has no interaction */
            () => root.vars.hasUserInteraction === false,
            /* test if the window object is trapped */
            () => root.vars.isCDPTrapped,
            /* test if the browser is a bot */
            () => ('webdriver' in navigator === true && navigator.webdriver !== false),
            /* playwright detection */
            () => (navigator.userAgent.includes("HeadlessChrome") || navigator.userAgent.includes("Playwright")),
            () => ('__playwright__binding__' in window),
            () => ('__pwInitScripts' in window)
        ];
        if (root.vars.delaystart && event && root.shuffle([...checks]).some(check => check(event))) {
            let eventType = Object.getPrototypeOf(event).constructor.name;
            if (root.vars.cdp && root.vars.isCDPTrapped) console.warn('CDP runtime detected for event:', eventType);
            if (root.vars.trigger) {
                // site operator has opted to trigger an event when a bot is detected
                let nonceresponse = root.vars.nonce && (event.srcElement.value === root.vars.nonceValue || event.srcElement === root.vars.noncefield);
                let detail = {
                    "hashcash_ip": root.vars.remote_addr,
                    "hashcash_time": root.vars.request_time,
                    "hashcash_ua": navigator.userAgent,
                    "hashcash_event_type": eventType, // attack type identifier
                    "hashcash_target": event.currentTarget.getAttribute('id') || 'unknown', // attack type identifier
                    "hashcash_bubbles":event.bubbles, // attack type identifier
                    "hashcash_trusted":event.isTrusted, // attack type identifier
                    "hashcash_url": window.location.href,
                    "hashcash_nonce": nonceresponse
                };
                window.dispatchEvent(new CustomEvent('plg_captcha_hashcash', {detail:detail}));
            }
            /* target element is not the listener and/or is not trusted (user initiated), so we do something else */
            if(root.vars.punish){
                // site operator has opted to inflict punishment on the bot attempting to trick the form
                root.vars.level = 32;
            } else {
                // site operator is a saint
                return;
            }
        } else if (root.vars.nonce && root.vars.noncefield) {
            // user passed bot check, remove nonce field
            root.container.form.removeChild(root.vars.noncefield);
            root.vars.noncefield = null; // Clear reference
        }
        root.vars.iterations = Math.min(Math.pow(100, root.vars.level), 1e9);
        
        // Create worker if not exists
        if (!root.worker) {
            root.worker = new Worker(Joomla.getOptions('system.paths').root + '/media/plg_captcha_hashcash/js/worker.js');
            root.worker.onmessage = function(e) {
                console.log('Worker message received:', e.data);
                const { type, count, error } = e.data;
                if (type === 'success') {
                    root.container.disabled = false;
                    root.container.value = count;
                    root.vars.success = true;
                    if(root.worker){
                        root.worker.terminate();
                        root.worker = null;
                    }
                    window.dispatchEvent(new Event('plg_captcha_hashcash_finished'));
                } else if (type === 'failed') {
                    console.error('HashCash failed to find nonce');
                    window.dispatchEvent(new CustomEvent('plg_captcha_hashcash_error', {detail: 'No solution found'}));
                } else if (type === 'error') {
                    console.error('Worker error:', error);
                } else if (type === 'progress') {
                    // Future use: show progress somewhere, e.g., a spinner or counter
                    // console.log('Progress:', count);
                }
            };
            root.worker.onerror = function(err) {
                console.error('Worker error:', err);
            };
        }

        const text = root.vars.remote_addr + root.vars.request_time;
        const payload = {
            text,
            level: root.vars.level,
            iterations: root.vars.iterations,
            hash_algorithm: root.vars.hash_algorithm
        };
        // console.log('Starting worker with payload:', payload);
        root.vars.isMining = true;
        root.worker.postMessage({
            type: 'start',
            payload
        });

        window.dispatchEvent(new Event('plg_captcha_hashcash_started'));
    }

    this.testWindowForCDPRuntime = function(window) {
        if(!root.vars.cdp) return false;
        let trapped = false;
        const e = new Error();
        window.Object.defineProperty(e, 'stack', {
            configurable: false,
            enumerable: false,
            get: () => { trapped = true; return null; }
        });
        try {
            throw e;
        } catch (_) {
            // CDP might still access stack here
        }
        return trapped;
    }

    // Shuffle function (Fisher-Yates)
    this.shuffle = function(array) {
        for (let i = array.length - 1; i > 0; i--) {
            const j = Math.floor(Math.random() * (i + 1));
            [array[i], array[j]] = [array[j], array[i]];
        }
        return array;
    }

    this.handleChange = function(e) { root.startCalculation(e); };
    
    this.destroy = function() {
        if (root.worker) {
            root.worker.terminate();
            root.worker = null;
        }
        root.vars.listener.removeEventListener('change', root.handleChange);
    };
    construct(options);
};

// DOM load event listener
window.addEventListener('DOMContentLoaded', ()=>{
    new HashCashClass(Joomla.getOptions('plg_captcha_hashcash'));
});