// worker.js

self.onmessage = async function(e) {
  const { type, payload } = e.data;

  if (type === 'start') {
    const { text, level, iterations, hash_algorithm } = payload;

    const pattern = new RegExp('^0{' + level + '}');
    let count = 0;
    const MAX_ITERATIONS = 1e9;

    while (iterations > 0 && count < MAX_ITERATIONS) {
      let hashBuffer;
      try {
        switch(hash_algorithm) {
          case 'PBKDF2':
            hashBuffer = await pbkdf2Subtle(text + count, 'JoomlaHashCashSalt', 1000, 'SHA-256', 32);
            break;
          case 'PBKDF264KB':
            hashBuffer = await pbkdf264kbSubtle(text + count);
            break;
          case '256':
          case '384':
          case '512':
            hashBuffer = await shaSubtle(text + count, hash_algorithm);
            break;
          default:
            self.postMessage({ type: 'error', error: 'Unsupported algorithm' });
            return;
        }

        const hashString = Array.from(new Uint8Array(hashBuffer))
          .map(b => b.toString(16).padStart(2, '0'))
          .join('');

        if (pattern.test(hashString)) {
          self.postMessage({ type: 'success', count });
          return;
        }

        count++;
        // Optional: send progress every ~100k iterations to avoid message flood
        if (count % 100000 === 0) {
          self.postMessage({ type: 'progress', count });
        }
      } catch (err) {
        count++;
      }
    }
    // If we exit the loop without finding a solution
    self.postMessage({ type: 'failed', reason: 'exhausted iterations' });
  }
};

// --- Hashing functions (copied verbatim, but use self.crypto) ---

async function shaSubtle(text, algo = '256') {
  const encoder = new TextEncoder();
  const textBuffer = encoder.encode(text);
  const hashBuffer = await self.crypto.subtle.digest('SHA-' + algo, textBuffer);
  return hashBuffer;
}

async function pbkdf2Subtle(text, salt, iterations, hash = 'SHA-256', keyLength = 32) {
  const encoder = new TextEncoder();
  const textBuffer = encoder.encode(text);
  const saltBuffer = encoder.encode(salt);
  const key = await self.crypto.subtle.importKey(
    'raw',
    textBuffer,
    { name: 'PBKDF2' },
    false,
    ['deriveBits']
  );
  const params = {
    name: 'PBKDF2',
    hash: hash,
    salt: saltBuffer,
    iterations: iterations
  };
  const derivedBits = await self.crypto.subtle.deriveBits(
    params,
    key,
    keyLength * 8
  );
  return derivedBits;
}

async function pbkdf264kbSubtle(text) {
  const encoder = new TextEncoder();
//   const pattern = new RegExp('^0{' + level + '}'); // Note: level not in scope here — we'll pass it if needed, or remove console.log

  const mem = new Uint8Array(64 * 1024);
  const inputBuffer = encoder.encode(text);
  const saltHash = await self.crypto.subtle.digest('SHA-256', inputBuffer);
  const salt = new Uint8Array(saltHash).slice(0, 16);

  for (let i = 0; i < 64; i++) {
    self.crypto.getRandomValues(mem);
    await self.crypto.subtle.digest('SHA-256', mem);
  }

  const baseKey = await self.crypto.subtle.importKey(
    'raw',
    inputBuffer,
    { name: 'PBKDF2' },
    false,
    ['deriveBits']
  );

  const derived = await self.crypto.subtle.deriveBits(
    {
      name: 'PBKDF2',
      salt: salt.buffer,
      iterations: 15000,
      hash: 'SHA-256'
    },
    baseKey,
    256
  );

  const hash = Array.from(new Uint8Array(derived))
    .map(b => b.toString(16).padStart(2, '0'))
    .join('');

  // Removed console.log — worker can't access DOM/console reliably in all contexts
  return derived;
}