<?php

/**
 * @copyright   Copyright (C) 2005 - 2013 Michael Richey. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\Content\IndexingAPI\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Cache\CacheController;
use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\Event\DispatcherInterface;
use Joomla\Component\Content\Site\Helper\RouteHelper as ContentRouteHelper;
use Joomla\Component\Contact\Site\Helper\RouteHelper as ContactRouteHelper;
use Joomla\Component\Newsfeeds\Site\Helper\RouteHelper as NewsfeedRouteHelper;
use Joomla\Component\Tags\Site\Helper\RouteHelper as TagRouteHelper;
use Joomla\Component\Weblinks\Site\Helper\RouteHelper as WeblinkRouteHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Factory;
use Joomla\CMS\Date\Date;

defined('_JEXEC') or die;


final class IndexingAPI extends CMSPlugin 
{
    protected $app;
    
    private $contexts = [
        'com_content.article'=> 'getArticleRoute',
        'com_contact.contact'=> 'getContactRoute',
        'com_newsfeeds.newsfeed'=> 'getNewsfeedRoute',
        'com_tags.tag'=> 'getTagRoute',
        'com_weblinks.weblink'=> 'getWeblinkRoute',
        'com_menus.item'=> 'getCustomRoute',
        'IndexingAPIExternalTrigger'=> 'getCustomRoute',
    ];

    protected $autoloadLanguage = true;

    /**
     * Cache instance.
     *
     * @var    CacheController
     * @since  1.5
     */
    private $cache;

    /**
     * Cache controller factory interface
     *
     * @var    CacheControllerFactoryInterface
     * @since  4.2.0
     */
    private $cacheControllerFactory;

    /**
     * Constructor
     *
     * @param   DispatcherInterface              $dispatcher                 The object to observe
     * @param   array   $config    An array that holds the plugin configuration
     *
     * @since   1.5
     */
    public function __construct(
        DispatcherInterface $dispatcher,
        array $config,
        CacheControllerFactoryInterface $cacheControllerFactory
    )
    {
        parent::__construct($dispatcher, $config);
        
        $this->cacheControllerFactory = $cacheControllerFactory;
    }

    /**
     * Event method triggered after saving a content item.
     *
     * @param   string   $context  The context of the content being saved.
     * @param   object   $article  The content item being saved.
     * @param   bool     $isNew    True if this is a new content item, false if it's an existing one.
     *
     * @return  void
     *
     * @since   5.0
     */
    public function onContentAfterSave($context, $article, $isNew)
    {
        if(!in_array($context, array_keys($this->contexts))) {
            return; // do nothing, this is not an article we can process
        }

        $allowed_levels = $this->params->get('publish_levels', [1,5]); // default to Public and Registered
        if(!in_array((int)$article->access, $allowed_levels)) {
            return; // do nothing, the article access level is not allowed
        }

        $contextarray = explode('.', $context);
        if(!(bool)$this->params->get('allcats',0) && !in_array($contextarray[0],['com_tags','IndexingAPIExternalTrigger'])) { // if not all categories
            // get the categories array
            $cats = $this->_getCatArray($this->params->get($contextarray[1].'categories', []));
            if(!in_array($article->catid, $cats)) { // if the category is not selected
                return; // do nothing, the category is not selected
            }
        }

        $published = isset($article->published) ? $article->published : $article->state;
        if($isNew && $published != 1) { 
            return; // do nothing, the article is not and was not published
        }

        $now = new Date();
        $publish_up = new Date($article->publish_up);
        if($publish_up->toUnix() > $now->toUnix()) {
            return; // do nothing, the article is not published yet
        }

        $robots = (bool)$this->params->get('robots', '0');
        if($robots){
            switch($context){
                case 'com_menus.item':
                    $metadataString = isset($article->params) ? $article->params : '{"robots":""}';
                    break;
                default:
                    $metadataString = isset($article->metadata) ? $article->metadata : '{"robots":""}';
                    break;
            }
            if($this->noindexRobots($metadataString)){
                return; // do nothing, the article is marked as noindex
            }
        }

        // initiate the indexing process
        $type = 'URL_UPDATED';
        if(!$isNew && $published != 1) {
            $type = 'URL_DELETED';
        }

        $url = $this->{$this->contexts[$context]}($article);
        if($url === false){
            return;
        }

        if($url === true){
            // custom route provided, with rawLink set to true
            // this means the plugin is triggered from a 3rd party extension
            // that cannot detec the hostname
            // so we need to use the hostname from $this->params
            $rooturi = $this->params->get('rooturi',false);
            if($rooturi === false || !strlen(trim($rooturi))) {
                $this->app->enqueueMessage(Text::_('PLG_CONTENT_INDEXINGAPI_NO_HOSTNAME'), 'Error');
                return;
            }
            // if the url does not begin with http, we need to add $rooturi and route the link
            if(strpos($article->link, 'http') !== 0) {
                $uri = $rooturi.htmlspecialchars_decode(Route::link('site',$article->link,true,0,false));
            } else {
                // otherwise we assume this is a complete, ready-to-submit url
                $uri = $article->link;
            }
        } else {
            $uri = Route::link('site',$url,true,0,true);
        }

        // if plugin settings are set to delete cache item, delete the cache item before initiating the indexing process
        // barring any external caching - this should force rendering of the page upon index update
        if($this->params->get('removecacheitem',0)) {
            $options = [
                'defaultgroup' => 'page'
            ];
            $this->cache = Factory::getContainer()->get(CacheControllerFactoryInterface::class)->createCacheController('page',$options);
            $key = md5(serialize([$uri]));
            if($this->cache->contains($key)) {
                $this->cache->remove($key);
                // $this->app->enqueueMessage(Text::sprintf('PLG_CONTENT_INDEXINGAPI_REMOVECACHEITEM_SUCCESS', $uri));
            // } else {
            //     $this->app->enqueueMessage(Text::sprintf('PLG_CONTENT_INDEXINGAPI_REMOVECACHEITEM_FAIL', $uri));
            }
        }

        $this->_initiateUpdate($uri, $type);
    }

    private function noindexRobots($jsonstring){
        $metadata = json_decode($jsonstring);
        $robotsMeta = explode(',',$metadata->robots);
        return in_array('noindex',$robotsMeta);
    }

    private function getCustomRoute($article)
    {
        // the provider must provide the route in a link attribute in the article object
        // this allows the plugin to be triggered from a 3rd party extension, as long as the route is provided
        if(isset($article->cli)){
            return true;
        }
        return $article->link;
    }

    private function getArticleRoute($article)
    {
        if((int)$article->state !== 1 || !in_array((int)$article->access,[1,5])) {
            return false;
        }
        return ContentRouteHelper::getArticleRoute(
            $article->id,
            $article->catid,
            $article->language
        );
    }

    private function getContactRoute($contact)
    {
        if((int)$contact->published !== 1 || !in_array((int)$contact->access,[1,5])) {
            return false;
        }
        return ContactRouteHelper::getContactRoute(
            $contact->id,
            $contact->catid,
            $contact->language
        );
    }

    private function getNewsfeedRoute($newsfeed)
    {
        if((int)$newsfeed->published !== 1 || !in_array((int)$newsfeed->access,[1,5])) {
            return false;
        }
        return NewsfeedRouteHelper::getNewsfeedRoute(
            $newsfeed->id,
            $newsfeed->catid,
            $newsfeed->language
        );
    }

    private function getTagRoute($tag)
    {
        if((int)$tag->published !== 1 || !in_array((int)$tag->access,[1,5])) {
            return false;
        }
        return TagRouteHelper::getTagRoute(
            $tag->id,
            $tag->catid,
            $tag->language
        );
    }

    private function getWeblinkRoute($weblink)
    {
        if((int)$weblink->state !== 1 || !in_array((int)$weblink->access,[1,5])) {
            return false;
        }
        return WeblinkRouteHelper::getWeblinkRoute(
            $weblink->id,
            $weblink->catid,
            $weblink->language
        );
    }

    /**
     * Get the categories array
     *
     * @param   array  $categories  The categories array.
     *
     * @return  array
     *
     * @since   5.0
     */
    protected function _getCatArray($categories)
    {
        $cats = [];
        foreach($categories as $cat) {
            $cats[] = $cat;
        }
        return $cats;
    }

    /**
     * Initiate the indexing process
     *
     * @param   Uri     $uri   The URI of the content item.
     * @param   string  $type  The type of indexing process.
     *
     * @return  void
     *
     * @since   5.0
     */
    protected function _initiateUpdate($uri, $type)
    {
        if($this->params->get('scheme',0,'BOOLEAN')) {
            $uri = str_replace('http://','https://',$uri);
        }
        $this->app->enqueueMessage(Text::sprintf('PLG_CONTENT_INDEXINGAPI_URL',$uri));
        if($this->params->get('google', 0)) {
            $jsonkey = $this->params->get('google_key', false);
            if(!$jsonkey || !strlen(trim($jsonkey))) {
                $this->app->enqueueMessage(Text::_('PLG_CONTENT_INDEXINGAPI_GOOGLE_NO_KEY_PRESENT'), 'Error');
                return;
            }
            // error_log('sending google indexing api request for url: '.$uri);
            $response = $this->_updateGoogle($uri, $type, $jsonkey);
            
            switch($response){
                case 200:
                    $this->app->enqueueMessage(Text::_('PLG_CONTENT_INDEXINGAPI_GOOGLE_RESPONSE_'.$response));
                    break;
                case 400:
                case 403:
                case 429:
                    $this->app->enqueueMessage(Text::_('PLG_CONTENT_INDEXINGAPI_GOOGLE_RESPONSE_'.$response), 'Error');
                    break;
                default:
                    $this->app->enqueueMessage(Text::sprintf('PLG_CONTENT_INDEXINGAPI_GOOGLE_RESPONSE_UNKNOWN',$response), 'Warning');
                    break;
            }
        }
        
        if($this->params->get('bing', 0)) {
            $key = $this->params->get('bing_key', false);
            if(!$key || !strlen($key)) {
                $this->app->enqueueMessage(Text::_('PLG_CONTENT_INDEXINGAPI_BING_NO_KEY_PRESENT'), 'Error');
                return;
            }
            // error_log('sending bing indexing api request for url: '.$uri);
            $response = $this->_updateBing($uri, $key);

            switch($response){
                case 200:
                case 202:
                    $this->app->enqueueMessage(Text::_('PLG_CONTENT_INDEXINGAPI_BING_RESPONSE_'.$response));
                    break;
                case 400:
                case 403:
                case 422:
                case 429:
                    $this->app->enqueueMessage(Text::_('PLG_CONTENT_INDEXINGAPI_BING_RESPONSE_'.$response), 'Error');
                    break;
                default:
                    $this->app->enqueueMessage(Text::sprintf('PLG_CONTENT_INDEXINGAPI_BING_RESPONSE_UNKNOWN',$response), 'Warning');
                    break;
            }
        }
    }

    /**
     * Update Google Indexing API
     *
     * @param   Uri     $uri      The URI of the content item.
     * @param   string  $type     The type of indexing process.
     * @param   string  $jsonkey  The Google API key.
     *
     * @return  void
     *
     * @since   5.0
     */
    protected function _updateGoogle($uri, $type, $jsonkey)
    {
        // require_once JPATH_PLUGINS . '/content/indexingapi/helpers/google.php';
        $content = json_encode(['url' => (string) $uri, 'type' => $type]);
        return GoogleIndexingAPI::getAccessToken($jsonkey, 'https://indexing.googleapis.com/v3/urlNotifications:publish', $content);
    }

    /**
     * Update Bing Indexing API
     *
     * @param   Uri     $uri  The URI of the content item.
     * @param   string  $key  The Bing API key.
     *
     * @return  void
     *
     * @since   5.0
     */
    protected function _updateBing($uri, $key)
    {
        // require_once JPATH_PLUGINS . '/content/indexingapi/helpers/bing.php';
        $rooturi = $this->params->get('rooturi',false);
        return BingIndexingAPI::indexNow($key, (string) $uri, $rooturi);
    }
}