<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  Content.InterLinked
 *
 * @copyright   (C) 2006 Michael Richey <https://www.richeyweb.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace RicheyWeb\Plugin\Content\InterLinked\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Factory;
use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\CMS\Router\Route;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\Component\Content\Site\Helper\RouteHelper;


// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Plugin to enable loading fluid gallery into content (e.g. articles)
 * This uses the {fluidgallery} syntax
 *
 * @since  1.5
 */
final class InterLinked extends CMSPlugin
{
    protected $app;
    protected $db;
    private $_cache = false;
    private $tooltips = false;
    private $associations = false;
    private $categories = array();
    private $replacements = array();

	/**
	 * Returns an array of events this subscriber will listen to.
	 *
	 * @return  array
	 */
	public static function getSubscribedEvents(): array
	{
		return [
			'onContentPrepare' => 'onContentPrepare',
		];
	}

    public function onContentPrepare($context, &$article, &$params, $page = 0) {

        // Don't run this plugin when the not viewing an article
        if ($context !== 'com_content.article') {
            return;
        }

        // only operate if the in site
        if(!$this->app->isClient('site')) {
            return true;
        }

        // Only execute if $article is an object and has a text property
        if (!is_object($article) || !property_exists($article, 'text') || is_null($article->text)) {
            return;
        }
        
        $this->_cache = Factory::getContainer()->get(CacheControllerFactoryInterface::class)->createCacheController('output',['defaultgroup'=>'plg_content_interlinked']);
        $this->_cache->setCaching(1);

        // retrieve category associations
        if(!$this->associations) {
            if($this->_cache->contains('associations')) {
                $this->associations = json_decode($this->_cache->get('associations'), true);
            } else {
                $this->associations = $this->_categoryAssociations();
                $this->_cache->store('associations', json_encode($this->associations));
            }
        }

        // if we aren't meant to operate in this category, just return
        if(!array_key_exists($article->catid??false, $this->associations)) {
            return;
        }

        if((bool)$this->params->get('tooltips', 0)) {
            // Add tooltips to the interlinked content
            HTMLHelper::_('bootstrap.tooltip','.hasTooltip');
            $this->tooltips = true;
        }

        // build keyword list, trimming whitespace
        // $keywords = explode(',',strtolower(str_replace(', ',',',$article->metakey)));
        $keywords = array_filter(array_map('trim', explode(',', strtolower(str_replace(', ', ',', $article->metakey)))), 'strlen');
        $keywords = array_unique($keywords);
        $keywords = array_combine($keywords,$keywords);
        if(!count($keywords)) {
            return;
        }

        $articleRegexp = $this->_articleRegexp($keywords);
        $queryRegexp = $this->_queryRegexp($keywords);
        if($this->_cache->contains($article->id)){
            $matches = json_decode($this->_cache->get($article->id),true);
        } else {
            $matches = $this->_getMatches((int)$article->id,$article->catid,$queryRegexp);
            $this->_cache->store($article->id, json_encode($matches));
        }
        if(!count($matches)) {
            return;
        }
        $links = $this->_removeLinks($article);
        $items = $this->_removeTitled($article);
        if((bool)$this->params->get('headers', 1)) {
            $headers = $this->_removeHeaders($article);
        }
        $this->_processMatches($matches,$article,$keywords);
        $this->_replaceLinks($article,$links,'placeholder');
        $this->_replaceLinks($article,$this->replacements,'linked');
        $this->_replaceTitled($article,$items,'imgplaceholder');
        if((bool)$this->params->get('headers', 1)) {
            $this->_replaceHeaders($article,$headers,'headerplaceholder');
        }
    }

    private function _categoryAssociations(): array
    {
        // Get the category associations for the article
        $associations = array();
        $assocParams = $this->params->get('associations', array());
        
        foreach($assocParams as $assoc){
            $associations[$assoc->category] = $assoc->interlinked;
        }
        return $associations;
    }

    private function _queryRegexp($keywords) {
        $keywordsQuoted = array();
        foreach ($keywords as $keyword) {
            $keyword = str_replace("'","\'",$keyword);
            $keywordsQuoted[] = preg_quote($keyword, '|');
        }
        return implode('|', $keywordsQuoted);
    }
    
    private function _articleRegexp($keywords) {
        // eliminates direct matches within anchor tags
        $regexp = '(?<=(^|))('.str_replace(array("'"),array("\'"),implode('|',$keywords)).')+(?![^>]*\<\/a>)'; 
        return $regexp;
    }

    private function _getMatches($id,$catid,$regexp) {
        // $query = 'SELECT id,title,alias,catid,language,metakey FROM #__content ';
        // $query.= 'WHERE state = 1 AND id != '.$id.' AND catid IN('.implode(',',$this->associations[$catid]).') AND (metakey RLIKE \''.$regexp.'\')';
        // $query.= 'ORDER BY publish_up DESC';
        $query = $this->db->createQuery(true)
            ->select(['id', 'title', 'alias', 'catid', 'language', 'metakey'])
            ->from($this->db->quoteName('#__content'))
            ->where($this->db->quoteName('state') . ' = 1')
            ->where($this->db->quoteName('id') . ' != ' . (int)$id)
            ->where($this->db->quoteName('catid') . ' IN (' . implode(',', array_map('intval', $this->associations[$catid])) . ')')
            ->where($this->db->quoteName('metakey') . ' RLIKE ' . $this->db->quote($regexp))
            ->order('publish_up DESC');
        // $this->db->setQuery($query);
        try {
            $this->db->setQuery($query);
            return $this->db->loadObjectList();
        } catch (\Exception $e) {
            $this->app->enqueueMessage('Database error in InterLinked plugin: ' . $e->getMessage(), 'error');
            return [];
        }
        // return $this->db->loadObjectList();
    }

    private function _removeTitled(&$article){
        // this removes any element with a src, alt, or title attribute
        $items = array();
        $matches = null;
        $count = 1;
        preg_match_all('/(?<titled><\w+\s(.*?)((src|alt|title)=["\'].*?["\']\s*)(.*?)>)/',$article->text,$matches);
        foreach($matches['titled'] as $item) {
            $replacekey = 'plg_content_interlinked_'.count($items).'_imgplaceholder';
            $items[] = $item;
            $article->text = str_replace($item,$replacekey,$article->text,$count);
        }
        return $items;
    }

    private function _replaceTitled(&$article,$items,$key){
        $count = 1;
        foreach($items as $index=>$item) {
            $replacekey = 'plg_content_interlinked_'.$index.'_'.$key;
            $article->text = str_replace($replacekey,$item,$article->text,$count);
        }
    }

    private function _removeHeaders(&$article){
        $headers = array();
        $matches = null;
        $count = 1;
        preg_match_all('/(?<headers><h(?<level>[1-6])(.*?)>(.*?)<\/h\k<level>>)/',$article->text,$matches);
        foreach($matches['headers'] as $header) {
            $replacekey = 'plg_content_interlinked_'.count($headers).'_headerplaceholder';
            $headers[] = $header;
            $article->text = str_replace($header,$replacekey,$article->text,$count);
        }
        return $headers;
    }

    private function _replaceHeaders(&$article,$headers,$key){
        $count = 1;
        foreach($headers as $index=>$header) {
            $replacekey = 'plg_content_interlinked_'.$index.'_'.$key;
            $article->text = str_replace($replacekey,$header,$article->text,$count);
        }
    }
    
    private function _removeLinks(&$article){
        $links = array();
        $matches = null;
        $count = 1;
        preg_match_all('/(?<links><a(.*?)\/a>)/',$article->text,$matches);
        foreach($matches['links'] as $link) {
            $replacekey = 'plg_content_interlinked_'.count($links).'_placeholder';
            $links[] = $link;
            $article->text = str_replace($link,$replacekey,$article->text,$count);
        }
        return $links;
    }
    
    private function _replaceLinks(&$article,$links,$key){
        $count = 1;
        foreach($links as $index=>$link) {
            $replacekey = 'plg_content_interlinked_'.$index.'_'.$key;
            $article->text = str_replace($replacekey,$link,$article->text,$count);
        }
    }
    
    private function _processMatches(&$matches,&$article,&$keywords) {
        foreach($matches as $match) {
            $mk = explode(', ',strtolower($match->metakey));
            $intersect = array_intersect($mk,$keywords);
            if(count($intersect)) {
                usort($intersect,function ($a, $b) {
                    return substr_count($b, ' ') - substr_count($a, ' ');
                });
                $keyword = array_shift($intersect);
                unset($keywords[$keyword]);
                $this->_replaceOne($match,$keyword,$article);
            }
        }
    }
    
    private function _replaceOne($match,$keyword,&$article) {
        $link = Route::_(RouteHelper::getArticleRoute($match->id.':'.$match->alias,$match->catid,$match->language));
        $regexp = '/(?<=(>|\s|\())'.preg_quote($keyword,'/').'(?=(\s|\.|\,|\!|\?|\)|<))/i';
        $matches = null;
        preg_match($regexp,$article->text,$matches);
        if(!count($matches)) {
            return;
        }
        $key = count($this->replacements);
        $linked = 'plg_content_interlinked_'.$key.'_linked';
        $targetValue = (int)$this->params->get('target', 1);
        $target = (bool)$targetValue ? '' : ' target="_blank" rel="noopener noreferrer"';
        $class = $this->params->get('class','plg_content_interlinked');
        $ariaLabel = ' aria-label="'.htmlspecialchars($match->title, ENT_QUOTES) . ($targetValue === '0' ? ' (' . Text::_('PLG_CONTENT_INTERLINKED_OPEN_NEW_WINDOW') . ')' : '').'"';
        $tooltip = '';
        if($this->tooltips) {
            $tooltip='title="'.str_replace(' ','&#32;',htmlspecialchars($match->title, ENT_QUOTES)).'"';
            $class.=' hasTooltip';
        //     $link = '<a href="'.$link.'" class="'.$class.'" '.$tooltip.$target.$ariaLabel.'>'.$matches[0].'</a>';
        // } else {
        //     $link = '<a href="'.$link.'" class="'.$class.'"'.$target.$ariaLabel.'>'.$matches[0].'</a>';
        }
        $link = '<a href="'.$link.'" data-asset-name="Content - InterLinked" class="'.$class.'" '.$tooltip.$target.$ariaLabel.'>'.$matches[0].'</a>';
        $this->replacements[] = $link;
        $article->text = preg_replace($regexp,$linked,$article->text,1);
    }
}