<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  Content.ogre
 *
 * @copyright   (C) 2006 Open Source Matters, Inc. <https://www.joomla.org>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace RicheyWeb\Plugin\Content\MiniGlobe\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use RicheyWeb\Plugin\Content\MiniGlobe\Extension\TagParser;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Plugin to enable loading modules into content (e.g. articles)
 * This uses the {dialoglightbox} syntax
 *
 * @since  1.5
 */
final class MiniGlobe extends CMSPlugin
{
    protected $app;
    private $_tagparser = false;
    private $globes = 0;
    private $options = [
        'bumpmap' => false,
        'clouds' => false,
        'textureImage' => 'media/plg_content_miniglobe/images/default.webp',
        'bumpmapImage' => 'media/plg_content_miniglobe/images/bumpmap.webp',
        'cloudsImage' => 'media/plg_content_miniglobe/images/clouds.webp',
        'brightness' => 1,
        'speed' => 0.005,
        'hoverStop' => false,
        'markerColor' => '#ff0000',
        'markerSize' => 1,
        'markers' => [],
        'markersAJAX' => false,
        'markersKML' => false,
        'markersAJAXInterval' => 0
    ];

    /**
     * Plugin that loads module positions within content
     *
     * @param   string   $context   The context of the content being passed to the plugin.
     * @param   object   &$article  The article object.  Note $article->text is also available
     * @param   mixed    &$params   The article params
     * @param   integer  $page      The 'page' number
     *
     * @return  void
     *
     * @since   1.6
     */
    public function onContentPrepare($context, &$article, &$params, $page = 0)
    {
        // Only execute in the site application
        if (!$this->app->isClient('site')) {
            return;
        }
        if(!strstr($article->text,'{miniglobe')) {
            return true;
        }
        if ($context == 'com_finder.indexer') {
            preg_replace('/{miniglobe(.*?)}/', '', $article->text);
            return true;
        }
        if(!$this->_tagparser){
            $this->_setup();
        }

        // strip out paragraph and div tags
        $regex = '/(<p.*?>)(\{miniglobe(.*?)\})(<\/p>)/';
        $article->text = preg_replace($regex, '$2', $article->text);

        // reset tags
        $this->_tagparser->tags = array();
        $this->_tagparser->parse($article->text);

        if (count($this->_tagparser->tags)) {
            foreach ($this->_tagparser->tags as $tag) {
                $replacement = $this->_setupGlobe($tag);
                $article->text = $tag->replace($article->text, $replacement);
            }
        }

    }

    private function _arrayToHtmlAttributes($array,$prefix=''){
        $attributes = array();
        foreach($array as $key=>$value){
            // escape quotes
            if(is_bool($value)){
                $value = $value ? 'true' : 'false';
            } elseif(is_array($value) || is_object($value)){
                $value = json_encode($value);
            } else {
                $value = (string)$value;
            }
            $value = htmlspecialchars($value, ENT_QUOTES);
            $attributes[] = $prefix.$key.'="'.$value.'"';
        }
        return implode(' ',$attributes);
    }

    private function _setupGlobe($tag){
        $id = 'miniglobe_'.$this->globes;
        $attributes = [];
        $class = $tag->getParam('class',false);
        foreach(array_keys($this->options) as $option){
            $value = $tag->getParam($option,false);
            if($value === false) continue;
            $attributes[strtolower($option)] = $value;
        }
        $html = '<div id="'.$id.'" class="plg_content_miniglobe'.($class?' '.$class:'').'" '.$this->_arrayToHtmlAttributes($attributes, 'data-').'></div>';
        $this->globes++;
        return $html;
    }

    private function _setup(){
        $this->_tagparser = new TagParser('miniglobe');
        $this->_tagparser->setOptions(array('selfclosing' => true));
        foreach($this->options as $option=>$default){
            switch($option){
                case 'bumpmap':
                case 'clouds':
                case 'hoverStop':
                    $this->options[$option] = $this->params->get($option,(bool)$default);
                    break;
                case 'markerSize':
                case 'brightness':
                case 'speed':
                    $this->options[$option] = $this->params->get($option,(float)$default);
                    break;
                case 'markerColor':
                    $this->options[$option] = $this->params->get($option,(string)$default);
                    break;
                // case 'textureImage':
                // case 'bumpmapImage':
                // case 'cloudsImage':
                case 'markers':
                    break;
                default:
                    $this->options[$option] = Uri::root(true).$this->params->get($option,$default);
                    break;
            }
        }
        $doc = $this->app->getDocument();
        $doc->addStyleSheet(Uri::root(true).'/media/plg_content_miniglobe/css/miniglobe.css');
        $doc->addScript(Uri::root(true).'/media/plg_content_miniglobe/js/three.min.js', ['version'=>'auto', 'defer' => 'defer']);
        $doc->addScript(Uri::root(true).'/media/plg_content_miniglobe/js/miniglobe.js', ['version'=>'auto', 'defer' => 'defer']);
        $doc->addScriptOptions('plg_content_miniglobe', $this->options);
    }

}
