/**
 * ModernClock - A modern, vanilla JavaScript analog clock library
 * Ground-up rewrite with zero dependencies
 * 
 * @version 2.0.0
 * @author Your Name
 * @license MIT
 */

class ModernClock {
  static version = '2.0.0';
  static instances = new Map();
  
  static config = {
    defaultRadius: 85,
    renderRadius: 100,
    defaultSkin: 'swissRail',
    smoothSecondHand: true,
    useRequestAnimationFrame: true
  };

  constructor(canvasId, options = {}) {
    this.canvasId = canvasId;
    this.canvas = document.getElementById(canvasId);
    
    if (!this.canvas) {
      throw new Error(`Canvas element with id "${canvasId}" not found`);
    }


    // Merge options with defaults
    this.options = {
      displayRadius: options.displayRadius || ModernClock.config.defaultRadius,
      skin: options.skin || ModernClock.config.defaultSkin,
      showSecondHand: options.showSecondHand !== false,
      showDate: options.showDate || false,
      showAMPM: options.showAMPM || false,
      gmtOffset: options.gmtOffset !== undefined ? parseFloat(options.gmtOffset) : null,
      smoothSecondHand: options.smoothSecondHand !== false,
      onTick: options.onTick || null,
      customSkin: options.customSkin || null,
      responsive: options.responsive !== false,
      useNTP: options.useNTP || false,
      useDateHeader: options.useDateHeader || null, // URL to fetch Date header from
      logo: options.logo || null
    };

    this.renderRadius = ModernClock.config.renderRadius;
    this.scale = this.options.displayRadius / this.renderRadius;
    this.animationFrameId = null;
    this.lastRenderTime = 0;
    this.isVisible = true;
    this.timeOffset = 0; // Offset from local time (for NTP sync)
    this.resizeObserver = null;
    this.logoImage = null;
    this.logoReady = false;

    this.init();
    this.observe(this.canvas);
    ModernClock.instances.set(canvasId, this);
  }

  observe(element){
    const observer = new IntersectionObserver((entries) => {
      entries.forEach(entry => {
        if (entry.isIntersecting) {
          this.isVisible = true;
          this.start();
        } else {
          this.isVisible = false;
          this.stop();
        }
      });
    });
    observer.observe(element);
  }

  init() {
    // Apply responsive styling
    if (this.options.responsive) {
      this.canvas.style.aspectRatio = '1';
      this.canvas.style.width = '100%';
      this.canvas.style.height = 'auto';
      this.canvas.style.maxWidth = '100%';
      this.canvas.style.display = 'block';
    }

    // Set initial canvas dimensions
    this.updateCanvasSize();

    // Get context and apply scaling
    this.ctx = this.canvas.getContext('2d');

    // Setup responsive resizing
    if (this.options.responsive && typeof ResizeObserver !== 'undefined') {
      this.resizeObserver = new ResizeObserver(() => {
        this.updateCanvasSize();
      });
      this.resizeObserver.observe(this.canvas.parentElement || this.canvas);
    }

    // Fetch NTP time if requested
    if (this.options.useNTP) {
      this.syncNTPTime();
    } else if (this.options.useDateHeader) {
      this.syncDateHeader();
    }

    // Load logo if provided
    if (this.options.logo) {
      this.loadLogo();
    }

    // Start the clock
    this.start();
  }

  reloadClock() {
    this.stop();
    this.init();
  }

  updateOption(key,value) {
    switch(key){
        case 'skin':
          this.options.skin = value;
          this.options.customSkin = null;
          break;
        case 'customSkin':
          this.options.customSkin = value;
          break;
        default:
          this.options[key] = value;
    }
    // this.reloadClock();
  }

  updateCanvasSize() {
    let size;
    
    if (this.options.responsive) {
      // Use container size
      const container = this.canvas.parentElement;
      if (container) {
        const rect = container.getBoundingClientRect();
        size = Math.min(rect.width, rect.height);
      } else {
        size = this.canvas.offsetWidth || this.options.displayRadius * 2;
      }
    } else {
      // Use fixed size
      size = this.options.displayRadius * 2;
    }

    // Set canvas resolution (actual pixels)
    const dpr = window.devicePixelRatio || 1;
    this.canvas.width = size * dpr;
    this.canvas.height = size * dpr;

    // Update scale
    this.scale = (size * dpr / 2) / this.renderRadius;

    // Reapply context scaling
    if (this.ctx) {
      this.ctx.setTransform(1, 0, 0, 1, 0, 0);
      this.ctx.scale(this.scale, this.scale);
    }
  }

  start() {
    if (this.animationFrameId) return; // Already running
    // const animate = () => {
    //   this.tick();
      // this.animationFrameId = requestAnimationFrame(animate);
    // };
    // animate();
    this.tick();
  }

  stop() {
    if (this.animationFrameId) {
      cancelAnimationFrame(this.animationFrameId);
      this.animationFrameId = null;
    }
  }

  destroy() {
    this.stop();
    if (this.resizeObserver) {
      this.resizeObserver.disconnect();
      this.resizeObserver = null;
    }
    ModernClock.instances.delete(this.canvasId);
  }

  async syncNTPTime() {
    // Check for cached offset first
    const cachedOffset = this.getNTPCookie();
    if (cachedOffset !== null) {
      this.timeOffset = cachedOffset;
      if (this.options.onNTPSync) {
        this.options.onNTPSync(this.timeOffset, true); // true = from cache
      }
      return;
    }

    // Multiple time API endpoints for redundancy
    const endpoints = [
      {
        url: 'https://timeapi.io/api/Time/current/zone?timeZone=UTC',
        parser: (data) => new Date(data.dateTime)
      },
      {
        url: 'https://worldtimeapi.org/api/timezone/Etc/UTC',
        parser: (data) => new Date(data.datetime || data.utc_datetime)
      },
      {
        url: 'http://worldtimeapi.org/api/timezone/Etc/UTC',
        parser: (data) => new Date(data.datetime || data.utc_datetime)
      }
    ];

    const maxRetries = 2; // Retry each endpoint up to 2 times
    let lastError = null;

    // Try each endpoint with retries
    for (const endpoint of endpoints) {
      for (let attempt = 0; attempt <= maxRetries; attempt++) {
        try {
          const response = await fetch(endpoint.url, {
            cache: 'no-store',
            mode: 'cors',
            signal: AbortSignal.timeout(5000) // 5 second timeout
          });

          if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
          }

          const data = await response.json();
          const serverTime = endpoint.parser(data);
          const localTime = new Date();

          // Calculate offset in milliseconds
          this.timeOffset = serverTime.getTime() - localTime.getTime();

          // Store in cookie (24 hour expiry)
          this.setNTPCookie(this.timeOffset);

          if (this.options.onNTPSync) {
            this.options.onNTPSync(this.timeOffset, false); // false = fresh sync
          }

          return; // Success! Exit function
        } catch (error) {
          lastError = error;
          
          // If it's a timeout or connection error, retry
          if (attempt < maxRetries) {
            // Wait before retry (exponential backoff: 500ms, 1000ms)
            await new Promise(resolve => setTimeout(resolve, 500 * (attempt + 1)));
            continue;
          }
          
          // Max retries reached for this endpoint, try next one
          console.warn(`ModernClock: Failed to sync with ${endpoint.url} after ${maxRetries + 1} attempts`);
        }
      }
    }

    // All endpoints failed
    console.warn('ModernClock: All NTP sync attempts failed, using local time', lastError);
    this.timeOffset = 0;

    if (this.options.onNTPError) {
      this.options.onNTPError(lastError || new Error('All time sync endpoints failed'));
    }
  }

  getNTPCookie() {
    const name = 'modernclock_ntp_offset=';
    // const decodedCookie = decodeURIComponent(document.cookie);
    // const cookies = decodedCookie.split(';');
    
    // for (let i = 0; i < cookies.length; i++) {
    //   let cookie = cookies[i].trim();
    //   if (cookie.indexOf(name) === 0) {
    //     const value = cookie.substring(name.length);
    //     return parseFloat(value);
    //   }
    // }
    const cookieStart = document.cookie.indexOf(name);
    if(cookieStart !== -1){
      return parseFloat(document.cookie.substring(cookieStart + name.length).split(';')[0]);
    }
    return null;
  }

  setNTPCookie(offset) {
    const expires = new Date();
    expires.setTime(expires.getTime() + (24 * 60 * 60 * 1000)); // 24 hours
    
    document.cookie = `modernclock_ntp_offset=${offset};expires=${expires.toUTCString()};path=/;SameSite=Lax`;
  }

  clearNTPCookie() {
    document.cookie = 'modernclock_ntp_offset=;expires=Thu, 01 Jan 1970 00:00:00 UTC;path=/;';
  }

  async syncDateHeader() {
    // Check for cached offset first
    const cachedOffset = this.getNTPCookie();
    if (cachedOffset !== null) {
      this.timeOffset = cachedOffset;
      if (this.options.onNTPSync) {
        this.options.onNTPSync(this.timeOffset, true); // true = from cache
      }
      return;
    }

    const url = this.options.useDateHeader;
    if (!url) return;

    try {
      // Make HEAD request for efficiency (only headers, no body)
      const response = await fetch(url, {
        method: 'HEAD',
        cache: 'no-store'
      });

      // Get Date header
      const dateHeader = response.headers.get('Date');
      if (!dateHeader) {
        throw new Error('No Date header in response');
      }

      // Parse server time from Date header
      const serverTime = new Date(dateHeader);
      const localTime = new Date();

      // Validate the parsed date
      if (isNaN(serverTime.getTime())) {
        throw new Error('Invalid Date header: ' + dateHeader);
      }

      // Calculate offset in milliseconds
      this.timeOffset = serverTime.getTime() - localTime.getTime();

      // Store in cookie (24 hour expiry)
      // console.log(`ModernClock: Calculated time offset: ${this.timeOffset}ms`);
      this.setNTPCookie(this.timeOffset);

      if (this.options.onNTPSync) {
        this.options.onNTPSync(this.timeOffset, false); // false = fresh sync
      }

      // console.log(`ModernClock: Date header sync successful. Offset: ${this.timeOffset}ms from ${url}`);
    } catch (error) {
      // console.warn('ModernClock: Date header sync failed, using local time', error);
      this.timeOffset = 0;

      if (this.options.onNTPError) {
        this.options.onNTPError(error);
      }
    }
  }

  tick(timestamp = 0) {
    if (!this.isVisible) return;
    // Check if canvas still exists in DOM
    if (!document.body.contains(this.canvas)) {
      this.destroy();
      return;
    }
    // fix re-display when hidden by interface (tabs, accordions)
    if(this.canvas.offsetParent === null){
      this.animationFrameId = requestAnimationFrame((ts) => this.tick(ts));
      return;
    }

    // Get current time with NTP offset
    const now = new Date(Date.now() + this.timeOffset);
    let hours, minutes, seconds, milliseconds;

    if (this.options.gmtOffset !== null) {
      // Use GMT + offset
      const offsetMs = this.options.gmtOffset * 3600000;
      const offsetDate = new Date(now.getTime() + offsetMs);
      hours = offsetDate.getUTCHours();
      minutes = offsetDate.getUTCMinutes();
      seconds = offsetDate.getUTCSeconds();
      milliseconds = offsetDate.getUTCMilliseconds();
    } else {
      // Use local time (with NTP correction)
      hours = now.getHours();
      minutes = now.getMinutes();
      seconds = now.getSeconds();
      milliseconds = now.getMilliseconds();
    }

    // Add smooth animation to second hand
    if (this.options.smoothSecondHand) {
      seconds += milliseconds / 1000;
    }

    this.render(hours, minutes, seconds);

    // Call user callback if provided
    if (this.options.onTick) {
      this.options.onTick(hours, minutes, seconds);
    }

    // Schedule next frame
    this.animationFrameId = requestAnimationFrame((ts) => this.tick(ts));
  }

  render(hours, minutes, seconds) {
    const skin = this.getSkin();
    
    // Clear canvas
    this.ctx.clearRect(0, 0, this.renderRadius * 2, this.renderRadius * 2);

    // Draw components in order
    if (skin.outerBorder) this.drawCircle(skin.outerBorder);
    if (skin.face) this.drawCircle(skin.face);
    
    this.drawIndicators(skin);
    
    // Draw logo if enabled and ready
    if (this.options.logo && this.logoReady) {
      this.drawLogo();
    }
    
    // Draw AM/PM indicator if enabled
    if (this.options.showAMPM) {
      this.drawAMPM(hours, skin);
    }
    
    if (skin.hourHand) this.drawHand((hours % 12 + minutes / 60) / 12, skin.hourHand);
    if (skin.minuteHand) this.drawHand((minutes + seconds / 60) / 60, skin.minuteHand);
    
    if (this.options.showSecondHand && skin.secondHand) {
      this.drawHand(seconds / 60, skin.secondHand);
      if (skin.secondDecoration) {
        this.drawHandDecoration(seconds / 60, skin.secondDecoration);
      }
    }

    if (skin.centerDot) this.drawCircle(skin.centerDot);
  }

  getSkin() {
    // Use custom skin if provided, otherwise use preset
    if (this.options.customSkin) {
      return this.options.customSkin;
    }
    
    const skinName = this.options.skin;
    if (!ModernClock.skins[skinName]) {
      console.warn(`Skin "${skinName}" not found, using default`);
      return ModernClock.skins[ModernClock.config.defaultSkin];
    }
    
    return ModernClock.skins[skinName];
  }

  drawCircle(style) {
    this.ctx.save();
    this.ctx.globalAlpha = style.alpha ?? 1;
    this.ctx.lineWidth = style.lineWidth ?? 1;

    this.ctx.beginPath();
    this.ctx.arc(
      this.renderRadius,
      this.renderRadius,
      style.radius,
      0,
      2 * Math.PI
    );

    if (style.fillColor) {
      this.ctx.fillStyle = style.fillColor;
      this.ctx.fill();
    }
    
    if (style.color && style.lineWidth) {
      this.ctx.strokeStyle = style.color;
      this.ctx.stroke();
    }

    this.ctx.restore();
  }

  drawIndicators(skin) {
    for (let i = 0; i < 60; i++) {
      const isLarge = i % 5 === 0;
      const indicator = isLarge ? skin.largeIndicator : skin.smallIndicator;
      
      if (indicator) {
        this.drawRadialLine(i / 60, indicator);
      }
    }
  }

  drawHand(angleFraction, style) {
    this.ctx.save();
    this.ctx.translate(this.renderRadius, this.renderRadius);
    this.ctx.rotate(Math.PI * (2 * angleFraction - 0.5));
    this.ctx.globalAlpha = style.alpha ?? 1;
    this.ctx.strokeStyle = style.color;
    this.ctx.lineWidth = style.lineWidth;
    this.ctx.lineCap = style.lineCap || 'round';

    this.ctx.beginPath();
    this.ctx.moveTo(style.startAt || 0, 0);
    this.ctx.lineTo(style.endAt, 0);
    this.ctx.stroke();

    this.ctx.restore();
  }

  drawHandDecoration(angleFraction, style) {
    this.ctx.save();
    this.ctx.translate(this.renderRadius, this.renderRadius);
    this.ctx.rotate(Math.PI * (2 * angleFraction - 0.5));
    this.ctx.globalAlpha = style.alpha ?? 1;
    this.ctx.lineWidth = style.lineWidth ?? 1;

    if (style.radius) {
      this.ctx.beginPath();
      this.ctx.arc(style.startAt || 0, 0, style.radius, 0, 2 * Math.PI);
      
      if (style.fillColor) {
        this.ctx.fillStyle = style.fillColor;
        this.ctx.fill();
      }
      
      if (style.color) {
        this.ctx.strokeStyle = style.color;
        this.ctx.stroke();
      }
    }

    this.ctx.restore();
  }

  drawRadialLine(angleFraction, style) {
    this.ctx.save();
    this.ctx.translate(this.renderRadius, this.renderRadius);
    this.ctx.rotate(Math.PI * (2 * angleFraction - 0.5));
    this.ctx.globalAlpha = style.alpha ?? 1;
    this.ctx.strokeStyle = style.color;
    this.ctx.lineWidth = style.lineWidth;
    this.ctx.lineCap = style.lineCap || 'butt';

    if (style.radius) {
      // Draw a circle at the position
      this.ctx.beginPath();
      this.ctx.arc(style.startAt, 0, style.radius, 0, 2 * Math.PI);
      
      if (style.fillColor) {
        this.ctx.fillStyle = style.fillColor;
        this.ctx.fill();
      }
      this.ctx.stroke();
    } else {
      // Draw a line
      this.ctx.beginPath();
      this.ctx.moveTo(style.startAt, 0);
      this.ctx.lineTo(style.endAt, 0);
      this.ctx.stroke();
    }

    this.ctx.restore();
  }

  drawAMPM(hours, skin) {
    // Determine AM or PM
    const isPM = hours >= 12;
    const text = isPM ? 'PM' : 'AM';
    
    // Get text style from skin or use defaults
    const textStyle = skin.ampmText || {
      color: skin.hourHand?.color || 'black',
      alpha: 0.8,
      fontSize: 10,
      fontFamily: 'Arial, sans-serif',
      fontWeight: 'bold'
    };

    this.ctx.save();
    this.ctx.translate(this.renderRadius, this.renderRadius);
    
    // Set text properties
    this.ctx.globalAlpha = textStyle.alpha;
    this.ctx.fillStyle = textStyle.color;
    this.ctx.font = `${textStyle.fontWeight} ${textStyle.fontSize}px ${textStyle.fontFamily}`;
    this.ctx.textAlign = 'center';
    this.ctx.textBaseline = 'middle';
    
    // Position at 6 o'clock, between center and hour mark
    // 6 o'clock is at 90 degrees (Math.PI/2 radians from top)
    const yPosition = 40; // Between center (0) and typical hour mark (60-80)
    
    this.ctx.fillText(text, 0, yPosition);
    
    this.ctx.restore();
  }

  async loadLogo() {
    if (!this.options.logo) return;
    
    const logo = this.options.logo;
    
    // Text type - instant, no loading needed
    if (logo.text) {
      this.logoReady = true;
      return;
    }
    
    // SVG type - convert to data URL for canvas compatibility
    if (logo.svg) {
      try {
        // Ensure SVG has proper namespace
        let svgString = logo.svg.trim();
        if (!svgString.includes('xmlns')) {
          svgString = svgString.replace('<svg', '<svg xmlns="http://www.w3.org/2000/svg"');
        }
        
        // Use base64 encoding for better browser compatibility
        const base64 = btoa(unescape(encodeURIComponent(svgString)));
        const dataUrl = `data:image/svg+xml;base64,${base64}`;
        
        this.logoImage = new Image();
        this.logoImage.onload = () => {
          this.logoReady = true;
        };
        this.logoImage.onerror = (e) => {
          console.warn('ModernClock: SVG logo failed to load', e);
          console.warn('SVG content:', svgString);
        };
        this.logoImage.src = dataUrl;
      } catch (error) {
        console.warn('ModernClock: Error loading SVG logo', error);
      }
      return;
    }
    
    // Image URL or data URL
    if (logo.src) {
      this.logoImage = new Image();
      this.logoImage.onload = () => {
        this.logoReady = true;
      };
      this.logoImage.onerror = (e) => {
        console.warn('ModernClock: Logo image failed to load from:', logo.src, e);
      };
      
      // Handle CORS for external images (not needed for data URLs)
      if (logo.src.startsWith('http') && 
          !logo.src.startsWith(window.location.origin) &&
          !logo.src.startsWith('data:')) {
        this.logoImage.crossOrigin = 'anonymous';
      }
      
      this.logoImage.src = logo.src;
    }
  }

  drawLogo() {
    if (!this.logoReady || !this.options.logo) return;
    
    const logo = this.options.logo;
    const skin = this.getSkin();
    
    this.ctx.save();
    this.ctx.translate(this.renderRadius, this.renderRadius);
    this.ctx.globalAlpha = logo.alpha ?? 1;
    
    if (logo.text) {
      // Draw text logo
      const textStyle = logo.style || {
        color: logo.color || skin.hourHand?.color || 'black',
        fontSize: logo.fontSize || 14,
        fontFamily: logo.fontFamily || 'Arial, sans-serif',
        fontWeight: logo.fontWeight || 'bold'
      };
      
      this.ctx.fillStyle = textStyle.color;
      this.ctx.font = `${textStyle.fontWeight} ${textStyle.fontSize}px ${textStyle.fontFamily}`;
      this.ctx.textAlign = 'center';
      this.ctx.textBaseline = 'middle';
      
      // Position at 12 o'clock
      const yPosition = logo.position?.y ?? -40;
      this.ctx.fillText(logo.text, logo.position?.x ?? 0, yPosition);
    } else if (this.logoImage) {
      // Draw image/SVG logo
      const w = logo.width || 30;
      const h = logo.height || 30;
      
      // Position at 12 o'clock (negative y = up)
      const x = logo.position?.x ?? 0;
      const y = logo.position?.y ?? -40;
      
      this.ctx.drawImage(this.logoImage, x - w/2, y - h/2, w, h);
    }
    
    this.ctx.restore();
  }

  // Update options dynamically
  updateOptions(newOptions) {
    const oldResponsive = this.options.responsive;
    const oldUseNTP = this.options.useNTP;
    const oldLogo = this.options.logo;
    
    Object.assign(this.options, newOptions);
    
    // If logo changed, reload it
    if (newOptions.logo !== undefined && newOptions.logo !== oldLogo) {
      this.logoReady = false;
      this.logoImage = null;
      if (newOptions.logo) {
        this.loadLogo();
      }
    }
    
    // If radius changed or responsive mode changed, update size
    if (newOptions.displayRadius || newOptions.responsive !== undefined) {
      // Handle responsive mode changes
      if (newOptions.responsive !== undefined && newOptions.responsive !== oldResponsive) {
        if (newOptions.responsive) {
          // Enable responsive mode
          this.canvas.style.aspectRatio = '1';
          this.canvas.style.width = '100%';
          this.canvas.style.height = 'auto';
          this.canvas.style.maxWidth = '100%';
          this.canvas.style.display = 'block';
          
          if (typeof ResizeObserver !== 'undefined' && !this.resizeObserver) {
            this.resizeObserver = new ResizeObserver(() => {
              this.updateCanvasSize();
            });
            this.resizeObserver.observe(this.canvas.parentElement || this.canvas);
          }
        } else {
          // Disable responsive mode
          if (this.resizeObserver) {
            this.resizeObserver.disconnect();
            this.resizeObserver = null;
          }
          this.canvas.style.aspectRatio = '';
          this.canvas.style.width = '';
          this.canvas.style.height = '';
        }
      }
      
      this.updateCanvasSize();
    }
    
    // Handle NTP sync changes
    if (newOptions.useNTP !== undefined && newOptions.useNTP && !oldUseNTP) {
      this.syncNTPTime();
    }
    
    // Handle Date header sync changes
    if (newOptions.useDateHeader !== undefined && newOptions.useDateHeader && !this.options.useDateHeader) {
      this.syncDateHeader();
    }
  }

  // Static method to find and create clocks from DOM
  static autoInit(selector = 'canvas[data-clock]') {
    const canvases = document.querySelectorAll(selector);
    
    canvases.forEach(canvas => {
      if (ModernClock.instances.has(canvas.id)) return; // Already initialized
      
      const dataset = canvas.dataset;
      const options = {
        displayRadius: dataset.radius ? parseInt(dataset.radius) : undefined,
        skin: dataset.skin || undefined,
        showSecondHand: dataset.showSeconds !== 'false',
        showAMPM: dataset.showAmpm === 'true',
        gmtOffset: dataset.gmtOffset ? parseFloat(dataset.gmtOffset) : undefined,
        smoothSecondHand: dataset.smoothSeconds !== 'false',
        responsive: dataset.responsive !== 'false',
        useNTP: dataset.useNtp === 'true'
      };

      new ModernClock(canvas.id, options);
    });
  }

  // Static method to get instance
  static getInstance(canvasId) {
    return ModernClock.instances.get(canvasId);
  }

  // Static method to destroy all instances
  static destroyAll() {
    ModernClock.instances.forEach(instance => instance.destroy());
    ModernClock.instances.clear();
  }

  // Static method to force NTP re-sync (clears cache)
  static forceNTPSync() {
    // Clear cookie
    document.cookie = 'modernclock_ntp_offset=;expires=Thu, 01 Jan 1970 00:00:00 UTC;path=/;';
    
    // Re-sync all clocks that use NTP
    ModernClock.instances.forEach(instance => {
      if (instance.options.useNTP) {
        instance.timeOffset = 0;
        instance.syncNTPTime();
      }
    });
  }
}

// Auto-initialize on DOM ready if data-auto-init is present
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', () => {
    if (document.querySelector('[data-clock-auto-init]')) {
      ModernClock.autoInit();
    }
    const scriptOptions = Joomla.getOptions('mod_coolclock');
    if(scriptOptions){
      Object.keys(scriptOptions).forEach(modId => {
        const options = scriptOptions[modId];
        const canvas = document.getElementById(modId + '_canvas');
        if (canvas) {
          new ModernClock(canvas.id, options);
        }
      });
    }
  });
} else {
  if (document.querySelector('[data-clock-auto-init]')) {
    ModernClock.autoInit();
  }
  const scriptOptions = Joomla.getOptions('mod_coolclock');
  if(scriptOptions){
    Object.keys(scriptOptions).forEach(modId => {
      const options = scriptOptions[modId];
      const canvas = document.getElementById(modId + '_canvas');
      if (canvas) {
        new ModernClock(canvas.id, options);
      }
    });
  }
}

// Export for module systems
if (typeof module !== 'undefined' && module.exports) {
  module.exports = ModernClock;
}

/**
 * ModernClock Skins
 * Skin definitions for ModernClock
 */

ModernClock.skins = {
  // Classic skins from original CoolClock
  swissRail: {
    outerBorder: { lineWidth: 1, radius: 95, color: 'black', alpha: 1 },
    smallIndicator: { lineWidth: 2, startAt: 89, endAt: 93, color: 'black', alpha: 1 },
    largeIndicator: { lineWidth: 4, startAt: 80, endAt: 93, color: 'black', alpha: 1 },
    hourHand: { lineWidth: 8, startAt: -15, endAt: 50, color: 'black', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 7, startAt: -15, endAt: 75, color: 'black', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 1, startAt: -20, endAt: 85, color: 'red', alpha: 1 },
    secondDecoration: { lineWidth: 1, startAt: 70, radius: 4, fillColor: 'red', color: 'red', alpha: 1 }
  },

  chunkySwiss: {
    outerBorder: { lineWidth: 4, radius: 97, color: 'black', alpha: 1 },
    smallIndicator: { lineWidth: 4, startAt: 89, endAt: 93, color: 'black', alpha: 1 },
    largeIndicator: { lineWidth: 8, startAt: 80, endAt: 93, color: 'black', alpha: 1 },
    hourHand: { lineWidth: 12, startAt: -15, endAt: 60, color: 'black', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 10, startAt: -15, endAt: 85, color: 'black', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 4, startAt: -20, endAt: 85, color: 'red', alpha: 1 },
    secondDecoration: { lineWidth: 2, startAt: 70, radius: 8, fillColor: 'red', color: 'red', alpha: 1 }
  },

  chunkySwissOnBlack: {
    face: { radius: 100, fillColor: '#000000', alpha: 1 },
    outerBorder: { lineWidth: 4, radius: 97, color: 'white', alpha: 1 },
    smallIndicator: { lineWidth: 4, startAt: 89, endAt: 93, color: 'white', alpha: 1 },
    largeIndicator: { lineWidth: 8, startAt: 80, endAt: 93, color: 'white', alpha: 1 },
    hourHand: { lineWidth: 12, startAt: -15, endAt: 60, color: 'white', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 10, startAt: -15, endAt: 85, color: 'white', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 4, startAt: -20, endAt: 85, color: 'red', alpha: 1 },
    secondDecoration: { lineWidth: 2, startAt: 70, radius: 8, fillColor: 'red', color: 'red', alpha: 1 }
  },

  fancy: {
    outerBorder: { lineWidth: 5, radius: 95, color: 'green', alpha: 0.7 },
    smallIndicator: { lineWidth: 1, startAt: 80, endAt: 93, color: 'black', alpha: 0.4 },
    largeIndicator: { lineWidth: 1, startAt: 30, endAt: 93, color: 'black', alpha: 0.5 },
    hourHand: { lineWidth: 8, startAt: -15, endAt: 50, color: 'blue', alpha: 0.7, lineCap: 'round' },
    minuteHand: { lineWidth: 7, startAt: -15, endAt: 92, color: 'red', alpha: 0.7, lineCap: 'round' },
    secondHand: { lineWidth: 10, startAt: 80, endAt: 85, color: 'blue', alpha: 0.3 },
    secondDecoration: { lineWidth: 1, startAt: 30, radius: 50, fillColor: 'blue', color: 'red', alpha: 0.15 }
  },

  machine: {
    outerBorder: { lineWidth: 60, radius: 55, color: '#dd6655', alpha: 1 },
    smallIndicator: { lineWidth: 4, startAt: 80, endAt: 95, color: 'white', alpha: 1 },
    largeIndicator: { lineWidth: 14, startAt: 77, endAt: 92, color: '#dd6655', alpha: 1 },
    hourHand: { lineWidth: 18, startAt: -15, endAt: 40, color: 'white', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 14, startAt: 24, endAt: 100, color: '#771100', alpha: 0.5, lineCap: 'round' },
    secondHand: { lineWidth: 3, startAt: 22, endAt: 83, color: 'green', alpha: 0 },
    secondDecoration: { lineWidth: 1, startAt: 52, radius: 26, fillColor: '#ffcccc', color: 'red', alpha: 0.5 }
  },

  classic: {
    face: { radius: 100, fillColor: '#E5ECF9', alpha: 1 },
    outerBorder: { lineWidth: 2, radius: 98, color: '#3366CC', alpha: 1 },
    smallIndicator: { lineWidth: 2, startAt: 89, endAt: 94, color: '#3366CC', alpha: 1 },
    largeIndicator: { lineWidth: 4, startAt: 83, endAt: 94, color: '#3366CC', alpha: 1 },
    hourHand: { lineWidth: 5, startAt: 0, endAt: 60, color: 'black', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 4, startAt: 0, endAt: 80, color: 'black', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 1, startAt: -20, endAt: 85, color: 'red', alpha: 0.85 },
    secondDecoration: { lineWidth: 3, startAt: 0, radius: 2, fillColor: 'black', color: 'black', alpha: 1 }
  },

  modern: {
    face: { radius: 100, fillColor: '#E5ECF9', alpha: 1 },
    outerBorder: { lineWidth: 2, radius: 98, color: '#3366CC', alpha: 1 },
    smallIndicator: { lineWidth: 5, startAt: 88, endAt: 94, color: '#3366CC', alpha: 1 },
    largeIndicator: { lineWidth: 5, startAt: 88, endAt: 94, color: '#3366CC', alpha: 1 },
    hourHand: { lineWidth: 8, startAt: 0, endAt: 60, color: 'black', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 8, startAt: 0, endAt: 80, color: 'black', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 5, startAt: 80, endAt: 85, color: 'red', alpha: 0.85 },
    secondDecoration: { lineWidth: 3, startAt: 0, radius: 4, fillColor: 'black', color: 'black', alpha: 1 }
  },

  simple: {
    face: { radius: 100, fillColor: '#E5ECF9', alpha: 1 },
    outerBorder: { lineWidth: 2, radius: 98, color: '#3366CC', alpha: 1 },
    smallIndicator: { lineWidth: 10, startAt: 90, endAt: 94, color: '#3366CC', alpha: 1 },
    largeIndicator: { lineWidth: 10, startAt: 90, endAt: 94, color: '#3366CC', alpha: 1 },
    hourHand: { lineWidth: 8, startAt: 0, endAt: 60, color: 'black', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 8, startAt: 0, endAt: 80, color: 'black', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 5, startAt: 80, endAt: 85, color: 'red', alpha: 0.85 },
    secondDecoration: { lineWidth: 3, startAt: 0, radius: 4, fillColor: 'black', color: 'black', alpha: 1 }
  },

  securephp: {
    outerBorder: { lineWidth: 100, radius: 0.45, color: '#669900', alpha: 0.3 },
    smallIndicator: { lineWidth: 2, startAt: 80, endAt: 90, color: 'green', alpha: 1 },
    largeIndicator: { lineWidth: 8.5, startAt: 20, endAt: 40, color: 'green', alpha: 0.4 },
    hourHand: { lineWidth: 3, startAt: 0, endAt: 60, color: 'black', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 2, startAt: 0, endAt: 75, color: 'black', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 1, startAt: -10, endAt: 80, color: 'blue', alpha: 0.8 },
    secondDecoration: { lineWidth: 1, startAt: 70, radius: 4, fillColor: 'blue', color: 'red', alpha: 1 }
  },

  Tes2: {
    outerBorder: { lineWidth: 4, radius: 95, color: 'black', alpha: 0.5 },
    smallIndicator: { lineWidth: 1, startAt: 10, endAt: 50, color: '#66CCFF', alpha: 1 },
    largeIndicator: { lineWidth: 8.5, startAt: 60, endAt: 70, color: '#6699FF', alpha: 1 },
    hourHand: { lineWidth: 5, startAt: -15, endAt: 60, color: 'black', alpha: 0.7, lineCap: 'round' },
    minuteHand: { lineWidth: 3, startAt: -25, endAt: 75, color: 'black', alpha: 0.7, lineCap: 'round' },
    secondHand: { lineWidth: 1.5, startAt: -20, endAt: 88, color: 'red', alpha: 1 },
    secondDecoration: { lineWidth: 1, startAt: 20, radius: 4, fillColor: 'blue', color: 'red', alpha: 1 }
  },

  Lev: {
    outerBorder: { lineWidth: 10, radius: 95, color: '#CCFF33', alpha: 0.65 },
    smallIndicator: { lineWidth: 5, startAt: 84, endAt: 90, color: '#996600', alpha: 1 },
    largeIndicator: { lineWidth: 40, startAt: 25, endAt: 95, color: '#336600', alpha: 0.55 },
    hourHand: { lineWidth: 4, startAt: 0, endAt: 65, color: 'black', alpha: 0.9, lineCap: 'round' },
    minuteHand: { lineWidth: 3, startAt: 0, endAt: 80, color: 'black', alpha: 0.85, lineCap: 'round' },
    secondHand: { lineWidth: 1, startAt: 0, endAt: 85, color: 'black', alpha: 1 },
    secondDecoration: { lineWidth: 2, startAt: 5, radius: 10, fillColor: 'black', color: 'black', alpha: 1 }
  },

  Sand: {
    outerBorder: { lineWidth: 1, radius: 70, color: 'black', alpha: 0.5 },
    smallIndicator: { lineWidth: 3, startAt: 50, endAt: 70, color: '#0066FF', alpha: 0.5 },
    largeIndicator: { lineWidth: 200, startAt: 80, endAt: 95, color: '#996600', alpha: 0.75 },
    hourHand: { lineWidth: 4, startAt: 0, endAt: 65, color: 'black', alpha: 0.9, lineCap: 'round' },
    minuteHand: { lineWidth: 3, startAt: 0, endAt: 80, color: 'black', alpha: 0.85, lineCap: 'round' },
    secondHand: { lineWidth: 1, startAt: 0, endAt: 85, color: 'black', alpha: 1 },
    secondDecoration: { lineWidth: 2, startAt: 5, radius: 10, fillColor: 'black', color: 'black', alpha: 1 }
  },

  Sun: {
    outerBorder: { lineWidth: 100, radius: 140, color: '#99FFFF', alpha: 0.2 },
    smallIndicator: { lineWidth: 300, startAt: 50, endAt: 70, color: 'black', alpha: 0.1 },
    largeIndicator: { lineWidth: 5, startAt: 80, endAt: 95, color: 'black', alpha: 0.65 },
    hourHand: { lineWidth: 4, startAt: 0, endAt: 65, color: 'black', alpha: 0.9, lineCap: 'round' },
    minuteHand: { lineWidth: 3, startAt: 0, endAt: 80, color: 'black', alpha: 0.85, lineCap: 'round' },
    secondHand: { lineWidth: 1, startAt: 0, endAt: 90, color: 'black', alpha: 1 },
    secondDecoration: { lineWidth: 2, startAt: 5, radius: 10, fillColor: 'black', color: 'black', alpha: 1 }
  },

  Tor: {
    outerBorder: { lineWidth: 10, radius: 88, color: '#996600', alpha: 0.9 },
    smallIndicator: { lineWidth: 6, startAt: -10, endAt: 73, color: 'green', alpha: 0.3 },
    largeIndicator: { lineWidth: 6, startAt: 73, endAt: 100, color: 'black', alpha: 0.65 },
    hourHand: { lineWidth: 4, startAt: 0, endAt: 65, color: 'black', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 3, startAt: 0, endAt: 80, color: 'black', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 1, startAt: -73, endAt: 73, color: 'black', alpha: 0.8 },
    secondDecoration: { lineWidth: 2, startAt: 5, radius: 10, fillColor: 'black', color: 'black', alpha: 1 }
  },

  Cold: {
    outerBorder: { lineWidth: 15, radius: 90, color: 'black', alpha: 0.3 },
    smallIndicator: { lineWidth: 15, startAt: -10, endAt: 95, color: 'blue', alpha: 0.1 },
    largeIndicator: { lineWidth: 3, startAt: 80, endAt: 95, color: 'blue', alpha: 0.65 },
    hourHand: { lineWidth: 4, startAt: 0, endAt: 65, color: 'black', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 3, startAt: 0, endAt: 80, color: 'black', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 1, startAt: 0, endAt: 85, color: 'black', alpha: 0.8 },
    secondDecoration: { lineWidth: 5, startAt: 30, radius: 10, fillColor: 'black', color: 'black', alpha: 1 }
  },

  Babosa: {
    outerBorder: { lineWidth: 100, radius: 25, color: 'blue', alpha: 0.25 },
    smallIndicator: { lineWidth: 3, startAt: 90, endAt: 95, color: '#3366CC', alpha: 1 },
    largeIndicator: { lineWidth: 4, startAt: 75, endAt: 95, color: '#3366CC', alpha: 1 },
    hourHand: { lineWidth: 4, startAt: 0, endAt: 60, color: 'black', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 3, startAt: 0, endAt: 85, color: 'black', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 12, startAt: 75, endAt: 90, color: 'red', alpha: 0.8 },
    secondDecoration: { lineWidth: 3, startAt: 0, radius: 4, fillColor: 'black', color: 'black', alpha: 1 }
  },

  Tumb: {
    outerBorder: { lineWidth: 105, radius: 5, color: 'green', alpha: 0.4 },
    smallIndicator: { lineWidth: 1, startAt: 93, endAt: 98, color: 'green', alpha: 1 },
    largeIndicator: { lineWidth: 50, startAt: 0, endAt: 89, color: 'red', alpha: 0.14 },
    hourHand: { lineWidth: 4, startAt: 0, endAt: 65, color: 'black', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 3, startAt: 0, endAt: 80, color: 'black', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 1, startAt: 0, endAt: 85, color: 'black', alpha: 0.8 },
    secondDecoration: { lineWidth: 5, startAt: 50, radius: 90, fillColor: 'black', color: 'black', alpha: 0.05 }
  },

  Stone: {
    outerBorder: { lineWidth: 15, radius: 80, color: '#339933', alpha: 0.5 },
    smallIndicator: { lineWidth: 2, startAt: 70, endAt: 90, color: '#FF3300', alpha: 0.7 },
    largeIndicator: { lineWidth: 15, startAt: 0, endAt: 29, color: '#FF6600', alpha: 0.3 },
    hourHand: { lineWidth: 4, startAt: 0, endAt: 65, color: 'black', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 3, startAt: 0, endAt: 75, color: 'black', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 1, startAt: 0, endAt: 85, color: 'black', alpha: 0.8 },
    secondDecoration: { lineWidth: 5, startAt: 50, radius: 90, fillColor: 'black', color: 'black', alpha: 0.05 }
  },

  Disc: {
    outerBorder: { lineWidth: 105, radius: 1, color: '#666600', alpha: 0.2 },
    smallIndicator: { lineWidth: 1, startAt: 58, endAt: 95, color: '#669900', alpha: 0.8 },
    largeIndicator: { lineWidth: 6, startAt: 25, endAt: 35, color: '#666600', alpha: 1 },
    hourHand: { lineWidth: 4, startAt: 0, endAt: 65, color: 'black', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 3, startAt: 0, endAt: 75, color: 'black', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 1, startAt: -75, endAt: 75, color: '#99CC00', alpha: 0.8 },
    secondDecoration: { lineWidth: 5, startAt: 50, radius: 90, fillColor: '#00FF00', color: 'green', alpha: 0.05 }
  },

  // NEW MODERN SKINS - These will set you apart

  minimal: {
    face: { radius: 100, fillColor: '#ffffff', alpha: 1 },
    outerBorder: { lineWidth: 1, radius: 99, color: '#e0e0e0', alpha: 1 },
    smallIndicator: { lineWidth: 1, startAt: 92, endAt: 96, color: '#cccccc', alpha: 1 },
    largeIndicator: { lineWidth: 2, startAt: 88, endAt: 96, color: '#666666', alpha: 1 },
    hourHand: { lineWidth: 6, startAt: -10, endAt: 50, color: '#333333', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 4, startAt: -10, endAt: 70, color: '#333333', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 2, startAt: -15, endAt: 80, color: '#ff4444', alpha: 1 },
    centerDot: { radius: 3, fillColor: '#333333', alpha: 1 }
  },

  dark: {
    face: { radius: 100, fillColor: '#1a1a1a', alpha: 1 },
    outerBorder: { lineWidth: 2, radius: 98, color: '#333333', alpha: 1 },
    smallIndicator: { lineWidth: 2, startAt: 90, endAt: 95, color: '#666666', alpha: 1 },
    largeIndicator: { lineWidth: 3, startAt: 85, endAt: 95, color: '#888888', alpha: 1 },
    hourHand: { lineWidth: 7, startAt: -12, endAt: 55, color: '#ffffff', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 5, startAt: -12, endAt: 75, color: '#ffffff', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 2, startAt: -18, endAt: 85, color: '#00d4ff', alpha: 1 },
    centerDot: { radius: 4, fillColor: '#00d4ff', alpha: 1 }
  },

  neon: {
    face: { radius: 100, fillColor: '#0a0a0a', alpha: 1 },
    outerBorder: { lineWidth: 3, radius: 97, color: '#ff00ff', alpha: 0.8 },
    smallIndicator: { lineWidth: 2, startAt: 90, endAt: 94, color: '#00ffff', alpha: 0.6 },
    largeIndicator: { lineWidth: 4, startAt: 85, endAt: 94, color: '#ff00ff', alpha: 0.8 },
    hourHand: { lineWidth: 8, startAt: -10, endAt: 50, color: '#ff00ff', alpha: 0.9, lineCap: 'round' },
    minuteHand: { lineWidth: 6, startAt: -10, endAt: 70, color: '#00ffff', alpha: 0.9, lineCap: 'round' },
    secondHand: { lineWidth: 2, startAt: -20, endAt: 85, color: '#ffff00', alpha: 1 },
    centerDot: { radius: 4, fillColor: '#ffffff', alpha: 1 }
  },

  ocean: {
    face: { radius: 100, fillColor: '#e6f3ff', alpha: 1 },
    outerBorder: { lineWidth: 4, radius: 96, color: '#0066cc', alpha: 0.8 },
    smallIndicator: { lineWidth: 2, startAt: 90, endAt: 94, color: '#0099cc', alpha: 0.7 },
    largeIndicator: { lineWidth: 5, startAt: 82, endAt: 94, color: '#0066cc', alpha: 0.9 },
    hourHand: { lineWidth: 9, startAt: -12, endAt: 55, color: '#003d7a', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 6, startAt: -12, endAt: 75, color: '#004d99', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 2, startAt: -20, endAt: 85, color: '#00ccff', alpha: 1 },
    secondDecoration: { lineWidth: 1, startAt: 70, radius: 5, fillColor: '#00ccff', color: '#00ccff', alpha: 1 }
  },

  sunset: {
    face: { radius: 100, fillColor: '#fff4e6', alpha: 1 },
    outerBorder: { lineWidth: 5, radius: 95, color: '#ff6b35', alpha: 0.9 },
    smallIndicator: { lineWidth: 2, startAt: 88, endAt: 93, color: '#ff8c42', alpha: 0.8 },
    largeIndicator: { lineWidth: 5, startAt: 80, endAt: 93, color: '#ff6b35', alpha: 1 },
    hourHand: { lineWidth: 9, startAt: -15, endAt: 50, color: '#d84315', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 7, startAt: -15, endAt: 70, color: '#f4511e', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 2, startAt: -20, endAt: 85, color: '#ffc107', alpha: 1 },
    centerDot: { radius: 4, fillColor: '#ffc107', alpha: 1 }
  },

  forest: {
    face: { radius: 100, fillColor: '#f1f8f4', alpha: 1 },
    outerBorder: { lineWidth: 4, radius: 96, color: '#2d5016', alpha: 1 },
    smallIndicator: { lineWidth: 2, startAt: 88, endAt: 93, color: '#558b2f', alpha: 0.8 },
    largeIndicator: { lineWidth: 5, startAt: 78, endAt: 93, color: '#2d5016', alpha: 1 },
    hourHand: { lineWidth: 9, startAt: -12, endAt: 55, color: '#1b3409', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 7, startAt: -12, endAt: 75, color: '#2d5016', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 2, startAt: -18, endAt: 85, color: '#8bc34a', alpha: 1 },
    secondDecoration: { lineWidth: 1, startAt: 70, radius: 5, fillColor: '#8bc34a', color: '#8bc34a', alpha: 1 }
  },

  luxury: {
    face: { radius: 100, fillColor: '#1a1a1a', alpha: 1 },
    outerBorder: { lineWidth: 8, radius: 92, color: '#d4af37', alpha: 1 },
    smallIndicator: { lineWidth: 1, startAt: 88, endAt: 90, color: '#d4af37', alpha: 0.6 },
    largeIndicator: { lineWidth: 4, startAt: 75, endAt: 90, color: '#d4af37', alpha: 1 },
    hourHand: { lineWidth: 10, startAt: -8, endAt: 50, color: '#d4af37', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 7, startAt: -8, endAt: 70, color: '#d4af37', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 2, startAt: -15, endAt: 82, color: '#ffd700', alpha: 1 },
    centerDot: { radius: 5, fillColor: '#d4af37', alpha: 1 }
  },

  retro: {
    face: { radius: 100, fillColor: '#fef5e7', alpha: 1 },
    outerBorder: { lineWidth: 6, radius: 94, color: '#d35400', alpha: 1 },
    smallIndicator: { lineWidth: 3, startAt: 88, endAt: 92, color: '#e67e22', alpha: 1 },
    largeIndicator: { lineWidth: 6, startAt: 80, endAt: 92, color: '#d35400', alpha: 1 },
    hourHand: { lineWidth: 10, startAt: -10, endAt: 50, color: '#a04000', alpha: 1, lineCap: 'round' },
    minuteHand: { lineWidth: 8, startAt: -10, endAt: 70, color: '#c0562f', alpha: 1, lineCap: 'round' },
    secondHand: { lineWidth: 3, startAt: -20, endAt: 85, color: '#e74c3c', alpha: 1 },
    secondDecoration: { lineWidth: 2, startAt: 70, radius: 6, fillColor: '#e74c3c', color: '#e74c3c', alpha: 1 }
  }
};
