class AdBlockReactor {
    constructor(options = {}) {
        this.vars = {
            timeout: 1000,
            baitClasses: 'pub_300x250 pub_300x250m pub_728x90 text-ad textAd text_ad text_ads text-ads text-ad-links',
            knownAdUrls: [
                'https://pagead2.googlesyndication.com/pagead/js/adsbygoogle.js',
                'https://securepubads.g.doubleclick.net/tag/js/gpt.js',
                'https://www.googletagservices.com/tag/js/gpt.js'
            ],
            actions: [],
            changeActions: [],
            lastReport: null,
            report: false,
            root: null,
            executed: false,
            useNetworkBait: true,
            useMutationObserver: true,
            useDomBait: true,
            randomBait: null,
            reload: true,
            mediaVersion: '1.0.0',
            worker: null
        };

        Object.assign(this.vars, options);

        // Randomize bait URL early (used later in test())
        if (this.vars.useNetworkBait && this.vars.knownAdUrls.length) {
            const index = Math.floor(Math.random() * this.vars.knownAdUrls.length);
            this.vars.baitUrl = this.vars.knownAdUrls[index];
        }

        // State
        this.detectionComplete = false;
        this.detectionTimeout = null;
        this.pendingResolve = null;
        this.currentBaitUrl = null;
        this.baitElement = null;
        this.observer = null;

        // Worker: only initialized here — no messages sent yet
        this.worker = null;
        this.workerReady = false;

        if (this.vars.useNetworkBait && this.vars.baitUrl && 'Worker' in window) {
            try {
                // const workerPath = this.vars.root ? `${this.vars.root}/media/plg_system_adblockreactor/js/worker.js` : '/media/plg_system_adblockreactor/js/worker.js';

                this.worker = new Worker(this.vars.worker);
                // this.worker = new Worker(workerPath);

                this.worker.onmessage = (e) => {
                    if (this.pendingResolve) {
                        const { blocked, details } = e.data;
                        // console.log('[Worker Detection]', { blocked, details, url: this.currentBaitUrl });
                        this.pendingResolve(blocked);
                        this.pendingResolve = null;
                    }
                };

                this.worker.onerror = (err) => {
                    // console.warn('[Worker Error]', err);
                    if (this.pendingResolve) {
                        this.pendingResolve(true); // Fail-safe: assume blocked
                        this.pendingResolve = null;
                    }
                    this.cleanupWorker();
                };

                this.workerReady = true;
            } catch (err) {
                // console.warn('[Worker Initialization Failed]', err);
                this.workerReady = false;
            }
        }

        // Kick off detection after construction
        this.test();
    }

    async test() {
        if (this.detectionComplete) return;

        if (this.vars.randomBait) {
            this.vars.baitClasses = this.vars.randomBait;
        }

        const checks = [];

        // DOM bait check
        if (this.vars.useDomBait) {
            this.baitElement = this.createBaitElement();
            document.body.appendChild(this.baitElement);

            if (this.vars.useMutationObserver) {
                this.observer = new MutationObserver(() => {
                    if (this.isBlocked(this.baitElement)) {
                        this.queueDetection(true);
                    }
                });
                this.observer.observe(this.baitElement, { attributes: true, childList: true, style: true });
            }

            checks.push(
                new Promise(resolve => {
                    setTimeout(() => {
                        resolve(this.isBlocked(this.baitElement));
                    }, this.vars.timeout);
                })
            );
        }

        // Network bait via worker — actual testing happens here
        if (this.vars.useNetworkBait && this.vars.baitUrl) {
            checks.push(this.checkNetworkBaitWithWorker());
        } else {
            checks.push(Promise.resolve(false));
        }

        // Local transparent GIF bait
        checks.push(this.checkNetworkBait());

        const results = await Promise.all(checks);
        const detected = results.some(r => r === true);

        this.queueDetection(detected);

        // Cleanup DOM-related stuff
        if (this.baitElement?.parentNode) this.baitElement.remove();
        if (this.observer) this.observer.disconnect();

        // Cleanup worker now that detection is done
        this.cleanupWorker();

        this.detectionComplete = true;
    }

    async checkNetworkBaitWithWorker() {
        return new Promise((resolve) => {
            if (!this.workerReady || !this.worker) {
                resolve(false);
                return;
            }

            this.pendingResolve = resolve;
            this.currentBaitUrl = this.vars.baitUrl;

            // This is where testing actually begins
            this.worker.postMessage({ url: this.currentBaitUrl });

            // Fallback if worker never responds
            setTimeout(() => {
                if (this.pendingResolve) {
                    // console.log('[Worker Timeout] Assuming blocked');
                    this.pendingResolve(true);
                    this.pendingResolve = null;
                }
            }, 5000);
        });
    }

    async checkNetworkBait() {
        return new Promise(resolve => {
            const img = new Image();
            img.src = this.vars.root + '/media/plg_system_adblockreactor/images/trans.gif?t=' + Date.now();

            img.onload = () => resolve(false);
            img.onerror = () => resolve(true);

            setTimeout(() => resolve(false), this.vars.timeout * 3);
        });
    }

    queueDetection(detected) {
        if (this.detectionTimeout) clearTimeout(this.detectionTimeout);

        this.detectionTimeout = setTimeout(() => {
            this.handleDetection(detected);
        }, 50);
    }

    async handleDetection(detected) {
        const current = detected;
        let event = null;
        switch(current) {
            case true:
                // console.log('[ABR] Ad blocker detected');
                event = 'SystemAdBlockReactorDetected';
                break;
            case false:
                // console.log('[ABR] No ad blocker detected');
                event = 'SystemAdBlockReactorCleared';
                break;
        }
        if(event) {
            window.dispatchEvent(new Event(event));
        }

        if (current !== this.vars.lastReport) {
            this.vars.lastReport = current;

            if (this.vars.changeActions.length) {
                this.execute(this.vars.changeActions);
            }

            if (this.vars.report) {
                try {
                    await this.report(current ? 1 : 0);
                    // console.log('[ABR] Report completed → refreshing page');
                    if(this.vars.reload){
                        setTimeout(() => window.location.reload(), 300);
                    }
                } catch (err) {
                    // console.warn('[ABR] Report failed:', err);
                }
            }
        }

        // if (detected && this.vars.actions.length) {
        //     setTimeout(() => this.execute(this.vars.actions), this.vars.timeout);
        // }

        // document.dispatchEvent(new CustomEvent('ABREvent', { detail: current }));
    }

    createBaitElement() {
        const el = document.createElement('div');
        el.style.position = 'fixed';
        el.style.left = '-1px';
        el.style.top = '-1px';
        el.style.width = '10px';
        el.style.height = '10px';
        el.style.zIndex = '-9999';
        el.style.pointerEvents = 'none';
        el.className = this.vars.baitClasses;

        const img = document.createElement('img');
        img.src = 'data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7';
        img.width = 10;
        img.height = 10;
        img.alt = '';
        el.appendChild(img);

        return el;
    }

    isBlocked(el) {
        if (!el) return false;
        const style = window.getComputedStyle(el);
        return (
            el.offsetHeight === 0 ||
            el.offsetWidth === 0 ||
            style.display === 'none' ||
            style.visibility === 'hidden' ||
            style.opacity === '0'
        );
    }

    execute(actions) {
        if (!actions || !actions.length) return;
        try {
            new Function(actions.join(';'))();
        } catch (e) {
            // console.error('AdBlockReactor action error:', e);
        }
    }

    report(value) {
        return new Promise((resolve, reject) => {
            const xhr = new XMLHttpRequest();
            const params = [
                'option=com_ajax',
                'format=raw',
                'group=system',
                'plugin=adblockreactor',
                'value=' + value
            ].join('&');

            const url = this.vars.root + '/?' + params + '&t=' + Date.now();

            xhr.open('GET', url, true);
            xhr.timeout = 5000;

            xhr.onload = () => {
                if (xhr.status >= 200 && xhr.status < 400) {
                    resolve(xhr.responseText);
                } else {
                    reject(new Error('HTTP ' + xhr.status));
                }
            };

            xhr.onerror = () => reject(new Error('Network error'));
            xhr.ontimeout = () => reject(new Error('Timeout'));

            xhr.send();
        });
    }

    cleanupWorker() {
        if (this.worker) {
            this.worker.terminate();
            this.worker = null;
            this.workerReady = false;
        }
    }
}

// Initialize on DOM ready
window.addEventListener('DOMContentLoaded', () => {
    const options = Joomla.getOptions?.('plg_system_adblockreactor') ||
                     Joomla.optionsStorage?.plg_system_adblockreactor ||
                     {};

    window.adblockreactor = new AdBlockReactor(options);
});
// window.addEventListener('SystemAdBlockReactorDetected', () => {
//     console.log('AdBlockReactor: Ad blocker detected');
//     // Custom actions on detection can be added here
// });
// window.addEventListener('SystemAdBlockReactorCleared', () => {
//     console.log('AdBlockReactor: Ad blocker cleared');
//     // Custom actions on clearance can be added here
// });