<?php

namespace RicheyWeb\Plugin\System\AdBlockReactor\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\Event\SubscriberInterface;
use Joomla\CMS\Event\Content\ContentPrepareEvent;

defined('_JEXEC') or die;

class Adblockreactor extends CMSPlugin implements SubscriberInterface
{
    protected $app;
    protected $doc;
    protected $wa;
    protected $autoloadLanguage = true;

    public static function getSubscribedEvents(): array
    {
        return [
            'onContentPrepare'        => 'onContentPrepare',
            'onBeforeCompileHead'         => 'onBeforeCompileHead',
            'onAjaxAdblockreactor' => 'onAjaxAdblockreactor',
        ];
    }

    public function onContentPrepare(ContentPrepareEvent $event) {
        // only operate in site application
        if (!$this->app->isClient('site')) {
            return;
        }

        $context = $event->getContext();
        if($context !== 'com_content.article' || !(bool)$this->app->getUserState('plg_system_adblockreactor.blocking', false)) {
            return; // only process articles
        }
        $article  = $event->getArgument('subject');  // &$article reference
        $params   = $event->getArgument('params');
        $page     = $event->getArgument('page', 0);


        // Get the configured reaction method
        $method = '_' . $this->params->get('content_method', 'nothing');

        if (method_exists($this, $method))
        {
            $this->$method($article);
        }

    }

    public function onBeforeCompileHead()
    {
        // Skip backend or non-HTML documents
        if (!$this->app->isClient('site'))
        {
            error_log('Skipping administrator client');
            return;
        }
        $this->doc = Factory::getDocument();
        $this->wa = $this->doc->getWebAssetManager();
        

        // Ignore specified user groups
        $ignoreGroups = $this->params->get('ignore', []);
        if (!empty($ignoreGroups))
        {
            $user       = $this->app->getIdentity();
            $userGroups = $user->getAuthorisedGroups();

            if (array_intersect((array)$ignoreGroups, $userGroups))
            {
                $this->updateUserAuthLevels(false);
                return;
            }
        }

        // Generate random bait class (always on for evasion — can expose toggle later)
        $randomBait = 'adbait-' . bin2hex(random_bytes(6));

        // Last report for change detection
        $lastReport = $this->app->getUserState('plg_system_adblockreactor.blocking');
        $lastReport = $lastReport === null ? null : (bool) $lastReport;

        $mediaVersion = $this->doc->getMediaVersion();
        $options = [
            'timeout'             => (int) $this->params->get('timeout', 100),
            'report'              => (bool) $this->params->get('report', false),
            'root'                => rtrim(Uri::root(), '/'),
            'fakeAdUrl'           => '/media/plg_system_adblockreactor/images/trans.gif',
            'useNetworkBait'      => (bool) $this->params->get('use_network_bait', true),
            'useMutationObserver' => (bool) $this->params->get('use_mutation_observer', true),
            'reload'              => (bool) $this->params->get('reload', true),
            'randomBait'          => $randomBait,
            // 'actions'             => $actions,
            // 'changeActions'       => $changeActions,
            'lastReport'          => $lastReport,
            'mediaVersion'        => $mediaVersion,
            'worker'              => rtrim(Uri::root(),'/') .'/media/plg_system_adblockreactor/js/worker' . $debug . '.js?' . $mediaVersion,
        ];

        // Register script and inject options

        $this->wa->registerAndUseScript(
            'plg_adblockreactor_detector',
            'media/plg_system_adblockreactor/js/abr.js',
            ['version' => 'auto', 'defer' => true]
        );

        $this->doc->addScriptOptions('plg_system_adblockreactor', $options);

    }

    /**
     * AJAX endpoint for reporting detection status
     */
    public function onAjaxAdblockreactor()
    {
        $value = $this->app->input->getInt('value', 0); // 1 = blocking, 0 = not

        $previous = $this->app->getUserState('plg_system_adblockreactor.blocking');
        $current  = (bool) $value;

        if ($previous === null || $previous !== $current) {
            if ((bool) $this->params->get('report', false)) {
                $this->updateUserAuthLevels($current);
            }
            $this->app->setUserState('plg_system_adblockreactor.blocking', $current);
        }

        // return application/javascript
        header('Content-Type: application/javascript');

        echo '/* OK */';
        $this->app->close();
    }

    private function _nothing(&$article): void
    {
        // Literally nothing
    }

    private function _truncate(&$article): void
    {
        $breakpoint = $this->params->get('content_breakpoint', 'readmore');

        if ($breakpoint === 'readmore' && $this->_hasReadMore($article))
        {
            $article->text = $article->introtext;
        }
        else
        {
            $counter = (int)$this->params->get('content_breakpoint_html_counter', 2) - 1;
            $tag     = $this->params->get('content_breakpoint_html', '<p>', 'RAW');
            $regex   = '/'.preg_quote($tag, '/').'/i';

            preg_match_all($regex, $article->text, $matches, PREG_OFFSET_CAPTURE);

            if (isset($matches[0][$counter]))
            {
                $pos = $matches[0][$counter][1];
                $article->text = substr($article->text, 0, $pos);
            }
        }

        $append = $this->params->get('content_append', '', 'RAW');
        if ($append)
        {
            $article->text .= $append;
        }
    }

    private function _redirect(&$article): void
    {
        $currentItemid = $this->app->getMenu()->getActive()?->id;
        $redirectItemid = (int)$this->params->get('content_redirect', 0);

        if (!$redirectItemid || $redirectItemid === $currentItemid)
        {
            return;
        }

        $url = Route::_('index.php?Itemid=' . $redirectItemid, false);

        $msgType = $this->params->get('content_redirect_message_type', 'message');
        $msg     = $this->params->get('content_redirect_message', Text::_('PLG_SYSTEM_ADBLOCKREACTOR_REDIRECT_MESSAGE'));

        if ($msg)
        {
            $this->app->enqueueMessage($msg, $msgType);
        }

        $this->app->redirect($url);
    }

    private function _counter(&$article): void
    {
        $max = (int)$this->params->get('content_counter_max', 3);
        $currentItemid = $this->app->getMenu()->getActive()?->id;
        $redirectItemid = (int)$this->params->get('content_redirect', 0);

        $counter = (int)$this->app->getUserState('plg_system_adblockreactor.counter', 0);

        if ($currentItemid !== $redirectItemid && $counter >= $max)
        {
            $this->_redirect($article);
            return;
        }

        if ($currentItemid !== $redirectItemid)
        {
            $counter++;
            $this->app->setUserState('plg_system_adblockreactor.counter', $counter);
        }

        // Expose to JS if you want a counter UI
        Factory::getDocument()->addScriptOptions('plg_system_adblockreactor_counter', [
            'count' => $counter,
            'max'   => $max
        ]);
    }

    private function _hasReadMore($article): bool
    {
        return !empty($article->fulltext);
    }

    /**
     * Dynamically add/remove the adACL viewing level using reflection
     */
    private function updateUserAuthLevels(bool $addLevel = true)
    {
        $adACL = (int) $this->params->get('adACL', 0);
        $clearACL = (int) $this->params->get('clearACL', 0);
        if ($adACL === 0 && $clearACL === 0)
        {
            return;
        }

        $user = $this->getApplication()->getIdentity();

        try
        {
            $reflection = new \ReflectionClass($user);
            $prop = $reflection->getProperty('authorisedLevels');
            $prop->setAccessible(true);

            $levels = $prop->getValue($user) ?? [];
            $levels = array_combine($levels, $levels); // reindex for search/unset

            if ($addLevel)
            {
                $add = $adACL;
                $remove = $clearACL;
            }
            else
            {
                $add = $clearACL;
                $remove = $adACL;
            }
            if ($add)
            {
                $levels[$add] = $add;
            }
            if ($remove)
            {
                unset($levels[$remove]);
            }
            $prop->setValue($user, array_values($levels));
        }
        catch (\ReflectionException $e)
        {
            // Silent fail — reflection unavailable (rare)
        }
    }
}