<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  System.bundler
 *
 * @copyright   (C) 2020 Michael Richey. <https://www.richeyweb.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace RicheyWeb\Plugin\System\Bundler\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Factory;
use Joomla\Event\SubscriberInterface;
use Joomla\Event\DispatcherInterface;
use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\Event\Priority;
use Joomla\CMS\Event\Model\SaveEvent;
use Joomla\CMS\Event\Extension\AfterInstallEvent;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * System plugin to package multiple frontend scripts into single files to reduce requests.
 *
 * @since  5.0.0
 */
final class Bundler extends CMSPlugin implements SubscriberInterface
{
    protected $app;
    protected $doc;
    protected $wa = null;
    protected $_cache;

    public function __construct(&$subject, $config)
    {
        $this->app = $this->getApplication();
        $this->debug = Factory::getConfig()->get('debug', false);
        parent::__construct($subject, $config);
    }

    /**
     * Returns an array of events this subscriber will listen to.
     *
     * @return  array
     *
     * @since   4.0.0
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'onBeforeCompileHead' => ['onBeforeCompileHead', Priority::MIN],
            // 'onInstallerAfterInstaller' => 'onInstallerAfterInstaller',
            'onExtensionAfterSave' => 'onExtensionAfterSave'
        ];
    }

	function onBeforeCompileHead() {
        // $this->app = $this->getApplication();
        $doc = $this->app->getDocument();
        if($this->app->isClient('administrator') || $doc->getType() != 'html') {
            return true;
        }
        $this->_cache = Factory::getContainer()->get(CacheControllerFactoryInterface::class)->createCacheController('output',['defaultgroup'=>'plg_system_bundler']);

        // first we grab the media version, as this is our cache key.
        // if the version changes, we re-generate the bundles and update the cache with scripts/styles to remove
        // if the version is the same, we skip the generation and just remove the scripts/styles from the document head
        $version = $doc->getMediaVersion();
        $removeScriptNames = [];
        $removeStyleNames = [];
        $this->checkCache($version, $removeScriptNames, $removeStyleNames);
        $bundles = $this->_getBundleFiles();
        $this->wa = $doc->getWebAssetManager();

        if(empty($removeScriptNames) && empty($removeStyleNames)) {
            $scripts = $this->wa->getAssets('script');
            $styles = $this->wa->getAssets('style');
            if(count($bundles['type']['script']) > 0) {
                $this->_collectAssets($scripts, 'script', $removeScriptNames, $bundles);
            }
            if(count($bundles['type']['style']) > 0) {
                $this->_collectAssets($styles, 'style', $removeStyleNames, $bundles);
            }
            $this->_cache->store([$removeScriptNames, $removeStyleNames], 'bundler_' . $version);
            $this->generateBundles();
        }

        foreach($removeScriptNames as $name) {
            try{
                $this->wa->disableAsset('script', $name);
            } catch(\Exception $e){
                if($this->debug) {
                    error_log("System - Bundler could not disable asset $name: " . $e->getMessage());
                }
            }
        }
        foreach($removeStyleNames as $name) {
            try{
                $this->wa->disableAsset('style', $name);
            } catch(\Exception $e){
                if($this->debug) {
                    error_log("System - Bundler could not disable asset $name: " . $e->getMessage());
                }
            }
        }

        $this->generateBundles('generateTags');
    }

    private function checkCache($version, &$removeScriptNames, &$removeStyleNames) {
        $this->_cache = Factory::getContainer()->get(CacheControllerFactoryInterface::class)->createCacheController('output',['defaultgroup'=>'plg_system_bundler']);
        if($this->_cache->contains('bundler_' . $version)) {
            $cached = $this->_cache->get('bundler_' . $version);
            $removeScriptNames = $cached[0];
            $removeStyleNames = $cached[1];
        }
    }

    // function onInstallerAfterInstaller(AfterInstallEvent $event) {
    //     $this->app = $this->getApplication();
    //     $this->doc = $this->app->getDocument();
    //     // so we need to clear the cache so bundles will be re-generated with new settings
    //     $version = $this->doc->getMediaVersion();
    //     $this->_cache = Factory::getContainer()->get(CacheControllerFactoryInterface::class)->createCacheController('output',['defaultgroup'=>'plg_system_bundler']);
    //     $this->_cache->clean('bundler_'.$version);
    //     $this->generateBundles();
    // }

    function onExtensionAfterSave(SaveEvent $event) {
        // $this->app = $this->getApplication();
        // $this->doc = $this->app->getDocument();
        $context = $event->getContext();
        $data = (array)$event->getData();
        if($context !== 'com_plugins.plugin' || $data['element'] !== 'bundler') {
            return;
        }
        // if we're here, we just saved the bundler plugin, 
        // so we need to clear the cache so bundles will be re-generated with new settings
        $version = $this->app->getDocument()->getMediaVersion();
        $this->_cache = Factory::getContainer()->get(CacheControllerFactoryInterface::class)->createCacheController('output',['defaultgroup'=>'plg_system_bundler']);
        $this->_cache->clean('bundler_'.$version);
        $this->generateBundles();
    }

    private function generateBundles($targetMethod = 'generateBundle') {
        // $doc = $this->app->getDocument();
        // $version = $doc->getMediaVersion();
        $version = $this->app->getDocument()->getMediaVersion();
        $bundles = $this->_getBundleFiles();
        foreach(['script','style'] as $type) {
            $types = ((array)$bundles)['type'];
            foreach((array)$types[$type] as $bundleName){
                $this->{$targetMethod}($bundleName,$bundles,$type,$version);
            }
        }        
    }

    private function generateTags($bundleName, $bundles, $type, $version) {
        $wa = $this->wa ?? $this->app->getDocument()->getWebAssetManager();
        $path = 'media/plg_system_bundler/' . $bundleName . '_' . $version . '.' . ($type == 'script' ? 'js' : 'css');
        if(!file_exists(JPATH_ROOT . '/' . $path)) {
            $this->generateBundle($bundleName, $bundles, $type, $version);
        }
        switch($type){
            case 'script':
                $wa->registerAndUseScript('plg_system_bundler-'.$bundleName, $path, ['version'=>'auto'], array_combine($bundles['attrs'][$bundleName],$bundles['attrs'][$bundleName]));
                break;
            case 'style':
                $wa->registerAndUseStyle('plg_system_bundler-'.$bundleName, $path, ['version'=>'auto']);
                break;
        }
    }

    private function generateBundle($bundleName, $bundles, $type, $version) {
        $this->clearBundleCache($bundleName);
        require_once __DIR__ . '/vendor/autoload.php';
        switch($type){
            case 'script':
                $minifier = new \MatthiasMullie\Minify\JS();
                break;
            case 'style':
                $minifier = new \MatthiasMullie\Minify\CSS();
                break;
        }
        foreach($bundles['files'] as $file=>$bundle) {
            if($bundle !== $bundleName) {
                continue;
            }

            $path = rtrim(JPATH_ROOT, '/') . '/' . ltrim($file, '/');
            try {
                $minifier->add('/** @origin '.$file." */");
                $minifier->add($path);
            } catch(\Exception $e){
                if($this->debug) {
                    error_log('System - Bundler could not add file ' . $file . ': ' . $e->getMessage());
                }
            }
        }
        $cachePath = JPATH_ROOT . '/media/plg_system_bundler/';
        $filename = $cachePath . $bundleName . '_' . $version . '.' . ($type == 'script' ? 'js' : 'css');
        $minified = $minifier->minify();
        if(!is_dir($cachePath)) {
            mkdir($cachePath, 0755, true);
        }
        file_put_contents($filename, $minified);
        //gzip'd version
        $filename = $filename.'.gz';
        file_put_contents($filename, gzencode($minified,9));
    }

    private function clearBundleCache($bundleName) {
        // delete all files in the cache folder for this plugin
        $path = JPATH_ROOT . '/media/plg_system_bundler/';
        if (is_dir($path)) {
            $files = glob($path . $bundleName.'_*'); // get all file names
            foreach ($files as $file) { // iterate files
                try{
                    unlink($file);
                } catch(\Exception $e){
                    if($this->debug) {
                        error_log('System - Bundler could not delete cache file ' . $file . ': ' . $e->getMessage());
                    }
                }
            }
        }        
    }

    private function _collectAssets($assets,$type,&$collection,$bundles){
        foreach($bundles['type'][$type] as $bundle) {
            foreach($assets as $asset){
                if(in_array($asset->getUri(), $bundles['bundle'][$bundle])) {
                    $collection[] = $asset->getName();
                }
            }
        }
    }

    private function _getBundleFiles() {
        $r = [
            "type" => ['script'=>[],'style'=>[]],
            "attrs" => [],
            "files" => [],
            "bundle" => []
        ];
        $bundles = $this->params->get('bundles', []);
        foreach($bundles as $bundle) {
            $r['attrs'][$bundle->name] = $bundle->attrs??[];
            $r['type'][$bundle->type][] = $bundle->name;
            $r['bundle'][$bundle->name] = [];
            foreach($bundle->files as $file) {
                $r['files'][$file->file] = $bundle->name;
                $r['bundle'][$bundle->name][] = $file->file;
            }
        }
        return $r;
    }

}
