<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  System.LinkCanonical
 *
 * @copyright   (C) 2007 Michael Richey <https://www.richeyweb.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace RicheyWeb\Plugin\System\LinkCanonical\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Event\DispatcherInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Filesystem\File;
use ReflectionClass;
use ReflectionProperty;


// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * RicheyWeb Link Canonical Plugin.
 *
 * @since  1.5
 */
final class LinkCanonical extends CMSPlugin
{
    protected $app;
    protected $doc;
    private $current;
    private $canonical = false;
    private $pagination = false;

    /**
     * This function is called after the application has been loaded
     * We use it to get the current URL and create the canonical URL
     */
    // public function onBeforeCompileHead(){
    public function onAfterDispatch(){
        $this->app = Factory::getApplication();
        if (!$this->app->isClient('site')) {
            return true;
        }
        $this->doc = $this->app->getDocument();
        if($this->doc->getType() != 'html') {
            return true;
        }
        // check if we should ignore this page
        $task = $this->app->input->get->getCmd('task',false);
        if($task !== false) {
            return true;
        }
        // check if we should ignore this component
        $option = $this->app->input->getCmd('option',false);
        $options = $this->getOptionsArray();
        if (in_array($option, $options)) {
            return true;
        }
        
        // first order of business - get the current URL
        $this->current = Uri::getInstance();

        // Obtaining the router allows us to get the current vars
        $router = $this->app->getRouter();
        $vars = array_filter(array_merge($router->getVars(), $this->current->getQuery(true)));
// error_log(print_r($vars,true));
        // now we create the canonical URL using the current vars
        $scheme = $this->getScheme();
        $url = $scheme . '://' . $this->getHost();
        
        // conditionally including the port in the URL
        $port = $this->getPort($scheme);
        if ($port) {
            $url .= ':' . $port;
        }

        $fixespath = JPATH_PLUGINS.'/system/linkcanonical/src/Fixes/LinkCanonical_'.$option.'.php';
        if(file_exists($fixespath)) {
            // error_log(print_r($vars,true));
            $fixes = 'RicheyWeb\Plugin\System\LinkCanonical\Fixes\LinkCanonical_'.$option;
            $route = Route::_($fixes::fix($vars,$this->current,$this->pagination),false);
            if($route) {
                $url .= $route;
            } else {
                $url .= Route::_('index.php?'.http_build_query($vars),false);
            }
        } else {
            $url .= Route::_('index.php?'.http_build_query($vars),false);
        }

        $urlParts = ['scheme', 'host', 'port', 'path', 'query'];
        $canonical = Uri::getInstance($url);
        // $query = $canonical->getQuery(true);
        $canonicalurl = $canonical->toString($urlParts);

        if($this->params->get('redirect',0,'BOOLEAN')) {
            $this->redirectCanonical($canonicalurl,$urlParts);
        }
        
        // store the canonical URL for later use
        $this->canonical = $canonicalurl;

        $this->addHeadLink();
        $this->addHeaderLink();
    }

    /**
     * Get the options array
     *
     * @return  array  The options array
     */
    private function getOptionsArray() {
        $ignore = (array)$this->params->get('ignore', [], 'ARRAY');
        $options = [];
        foreach($ignore as $item) {
            $options[] = trim($item->option);
        }
        return $options;
    }

    /**
     * Redirect the user to the canonical URL
     * after matching certain criteria
     *
     * @param   string  $url  The canonical URL
     */
    private function redirectCanonical($url,$urlParts) {
        // never redirect the homepage
        if($this->current->toString() == Uri::root()) {
            return;
        }

        // compare the current URL to the canonical URL, if no match - redirect
        $current = $this->current->toString($urlParts);
        // if the current url doesn't begin with the canonical url, redirect; this saves redirects on 
        // untracked query vars because the canonical will still be correct, but the current url will be different
        if(strpos($current,$url) !== 0) {
            $this->app->redirect($url);
        }
    }

    /**
     * Get the port for the URL
     *
     * @param   string  $scheme  The scheme of the URL
     *
     * @return  mixed  The port number or false
     */
    private function getPort($scheme) {
        // we're checking that the port matches the scheme.  If it doesn't, we need to include it in the URL
        $port = $this->current->getPort($scheme);
        switch($scheme) {
            case 'http':
                if((int)$port == 80) {
                    return false;
                }
                break;
            case 'https':
                if((int)$port == 443) {
                    return false;
                }
                break;
        }
        return $port;
    }

    /**
     * Get the host for the URL
     *
     * @return  string  The host
     */
    private function getHost() {
        $param = $this->params->get('host','');
        if(!strlen(trim($param))) {
            return $this->current->getHost();
        }
        return trim($param);
    }

    /**
     * Get the scheme for the URL
     *
     * @return  string  The scheme
     */
    private function getScheme() {
        $param = $this->params->get('scheme','0');
        if($param == '0') {
            return $this->current->getScheme();
        }
        return $param;
    }

    /**
     * Add the canonical link to the head
     */
    private function addHeadLink(){
        $headData = $this->doc->getHeadData();
        $canonicalKey = $this->findCanonicalKey($headData['links']);
        if($canonicalKey !== false){
            $override = (bool)$this->params->get('override',0,'INTEGER');
            if(!$override) {
                return;
            }
            unset($this->doc->_links[$canonicalKey]);
        }
        $this->doc->addHeadLink($this->canonical, 'canonical','rel',['data-asset-name'=>'System - Link Canonical']);
        if(is_array($this->pagination) && count($this->pagination)){
            error_log(print_r($this->pagination,true));
            if($this->pagination[0]??0 > 0) {
                $this->doc->addHeadLink($this->appendPagination($this->canonical,$this->pagination[0]), 'prev','rel',['data-asset-name'=>'System - Link Canonical']);
            }
            if($this->pagination[2] === 2) {
            //  this is so annoying - you can't have 2 different rels pointing to the same url...so I have to insert this one as a
            //  custom tag...FML
            //  $this->doc->addHeadLink($this->canonical, 'prev','rel',['data-asset-name'=>'System - Link Canonical']);
                $this->doc->addCustomTag('<link href="'.$this->appendPagination($this->canonical,false).'" rel="prev" data-asset-name="System - Link Canonical" />');
            }
            if(isset($this->pagination[1])) {
                $this->doc->addHeadLink($this->appendPagination($this->canonical,$this->pagination[1]), 'next','rel',['data-asset-name'=>'System - Link Canonical']);
            }
        }
    }

    private function appendPagination($url,$var){
        // die(print_r($this->pagination));
        $append = [];
        $input = $this->app->getInput();
        if(isset($this->pagination[3])){
            foreach($this->pagination[3] as $key=>$filter){
                $value = $input->get($key,null,$filter);
                if($value !== null){
                    $append[] = $key.'='.rawurlencode($value);
                }
            }
        }
        $o = parse_url($url.'?'.implode('&',$append));
        $user = isset($o['user']) ? $o['user'] : null;
        $pass = isset($o['pass']) ? $o['pass'] : null;
        $userpass = strlen($user.$pass)?$user.':'.$pass.'@':'';
        $port = isset($o['port']) ? ':'.$o['port'] : null;
        parse_str(($o['query']??''),$query);
        unset($query['start']);
        if(!count($query)) {
            return $o['scheme'].'://'.$userpass.$o['host'].$port.$o['path'].($var?'?start='.(int)$var:'');
        }
        if($var){
            $query['start'] = (int)$var;
        }
        $newquery = http_build_query($query);
        return $o['scheme'].'://'.$userpass.$o['host'].$port.$o['path'].'?'.$newquery;
    }

    private function findCanonicalKey($links){
        foreach($links as $key => $link){
            if(!isset($link['relation'])) {
                continue;
            }
            if($link['relation'] === 'canonical'){
                return $key;
            }
        }
        return false;
    }

    /**
     * Add the canonical link to the header
     * The preload manager link method is private, so we need to use reflection to access it
     * but that's super easy, barely an inconvenience
     * you can't stop the signal 🖕
    */
    private function addHeaderLink(){
        $preloadManager = $this->doc->getPreloadManager();
        $preloadReflection = new ReflectionClass($preloadManager);
        $linksMethod = $preloadReflection->getMethod('link');
        $linksMethod->setAccessible(true);
        $linksMethod->invokeArgs($preloadManager, [$this->canonical,'canonical']);
    }
}
