<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  System.adminexile
 *
 * @copyright   (C) 2020 Michael Richey. <https://www.richeyweb.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace RicheyWeb\Plugin\System\Offline\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Event\SubscriberInterface;
use Joomla\Event\DispatcherInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use RicheyWeb\Plugin\System\Offline\Extension\SimpleCIDR;
use RicheyWeb\Plugin\System\Offline\Extension\SimpleCIDR6;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * System plugin to add additional security features to the administrator interface.
 *
 * @since  5.0.0
 */
final class Offline extends CMSPlugin
{
    protected $app;
    private $_ipv;
    private $_ip;
    private $_key;

    /**
     * Returns an array of events this subscriber will listen to.
     *
     * @return  array
     *
     * @since   4.0.0
     */
    public static function getSubscribedEvents(): array
    {
        return ['onAfterInitialise' => 'onAfterInitialise'];
    }

	public function onAfterInitialise() {
        $this->app = Factory::getApplication();
        if($this->app->isClient('administrator')) {
            // don't run in the admin interface
            return;
        }
        if($this->app->getConfig()->get('offline') != 1) {
            // site is not offline
            return;
        }
        if($this->app->getIdentity()->id) {
            // user is logged in, already authenticated
            $this->_online();
            return;
        }
        $this->_ip = $this->_getIP();
        $this->_ipv = strstr($this->_ip,':' ) ? 'SimpleCIDR6' : 'SimpleCIDR';
        if($this->_whitelisted()) {
            $this->_online();
            return;
        }
        if($this->_blacklisted()) {
            return;
        }
        $this->_key = $this->params->get('key', 'true');
        $input = $this->app->input;
        $cookievalue = $input->cookie->get('online',false);
        if($cookievalue === $this->_key) {
            $this->_online();
            return true;
        }
        $this->_key = $this->params->get('key', 'true');
        if($this->_keyauth()) {
            $this->_authorize();
            $this->_online();
            return;
        }
    }

    private function _online(){
        $config = Factory::getConfig();
        $config->set('offline',0); 
    }
    function _authorize(){
        $config = Factory::getConfig();
        $input = $this->app->input;
        $input->cookie->set('online',$this->_key,0,$config->get('cookie_path'),$config->get('cookie_domain'),false,true);
    }

    private function _keyauth() {
        if(Uri::getInstance()->hasVar('online')){
            return $this->app->input->get('online','true') === $this->_key;
        }
        return false;
    }

    private function _whitelisted() {
		return $this->_listcontains('whitelist');
    }

    private function _blacklisted() {
		if ($match = $this->_listcontains('blacklist'))
		{
			return true;
		}
		return false;
    }

	private function _listcontains($source) {
        $list = $this->params->get($source,[]);
        if (empty((array)$list))
        {
            return false;
        }
		foreach ((array) $list as $item)
		{
			$item->address = trim($item->address);
			$item->address = strtolower($item->address);
			$net46 = preg_match_all('/:/', $item->address) ? 'SimpleCIDR6' : 'SimpleCIDR';
			// reasons to skip
			if (
				($this->_ipv !== $net46) || // don't bother testing the wrong ip version
				($this->_ipv === 'SimpleCIDR' && $item->netmask > 32) // invalid netmask for IPv4
			)
			{
				continue;
			}
			$net = trim($item->address) . '/' . trim($item->netmask);
            $classpath = 'Joomla\\Plugin\\System\\Offline\\Extension\\'.$net46;
			$test = $classpath::getInstance($net);
			$result = $test->contains($this->_ip);
			if ($result !== false)
			{
				return $item->address . '/' . $item->netmask;
			}
		}
		return false;
	}

    private function _getIP() {
        return getenv('HTTP_CLIENT_IP')?:
        getenv('HTTP_X_FORWARDED_FOR')?:
        getenv('HTTP_X_FORWARDED')?:
        getenv('HTTP_FORWARDED_FOR')?:
        getenv('HTTP_FORWARDED')?:
        getenv('REMOTE_ADDR');
    }

}