class SessionKeeperClient {
    constructor() {
        this.options = Joomla.getOptions('plg_system_sessionkeeper');
        if (!this.options) {
            console.warn('SessionKeeper: No options found');
            return;
        }
        this.Text = Joomla.Text;
        this.uiHandler = null;
        this.worker = null;
        this.port = null;
        this.broadcast = null;

        this.setupActivityDetection();
        this.initWorker();

        // Expose globally early
        window.SessionKeeper = this;

        // Signal readiness to any waiting handlers
        window.dispatchEvent(new CustomEvent('SessionKeeperReady', {
            detail: { client: this }
        }));
    }

    setupActivityDetection() {
        // 1. Modern, lightweight XHR wrapper (replaces hijackxhr.js)
        (function() {
            const originalOpen = XMLHttpRequest.prototype.open;
            XMLHttpRequest.prototype.open = function(method, url) {
                this.addEventListener('loadend', function() {
                    if (this.responseURL) {
                        window.dispatchEvent(new CustomEvent('SessionKeeperXHROpen', {
                            detail: this.responseURL
                        }));
                    }
                });
                originalOpen.apply(this, arguments);
            };
        })();

        // 2. PerformanceObserver for fetch + many XHR
        const observer = new PerformanceObserver((list) => {
            for (const entry of list.getEntries()) {
                if (entry.initiatorType === 'fetch' || entry.initiatorType === 'xmlhttprequest') {
                    window.dispatchEvent(new CustomEvent('SessionKeeperXHROpen', {
                        detail: entry.name
                    }));
                }
            }
        });
        observer.observe({ entryTypes: ['resource'] });

        // 3. Unified listener: filter + forward to worker
        window.addEventListener('SessionKeeperXHROpen', (e) => {
            this.reportActivity(e.detail);
        });

        // disconnect observer on unload
        window.addEventListener('unload', () => {
            observer.disconnect();
        });
    }

    reportActivity(url) {
        try {
            const parsed = new URL(url, location.href);
            if (parsed.hostname !== location.hostname) return;
            console.log('[SessionKeeper] Activity reported from: ' + url);
            // Always tell our own worker
            this.port.postMessage({ type: 'activity' });

            // In dedicated mode, also broadcast to siblings
            if (!this.isShared && this.broadcast) {
                this.broadcast.postMessage('activity');
            }
        } catch (err) {}
    }

    initWorker() {
        // Get paths from Joomla (always available)
        const paths = Joomla.getOptions('system.paths') || {};
        const rootFull = paths.rootFull || window.location.origin + '/';

        // Clean trailing slash (your replace was spot-on)
        const base = rootFull.replace(/\/$/, '');

        // Absolute URL to worker script
        const workerUrl = base + '/media/plg_system_sessionkeeper/js/worker.js';

        if ('SharedWorker' in window) {
            try {
                this.worker = new SharedWorker(workerUrl, 'SessionKeeperSharedWorker');
                this.port = this.worker.port;

                // CRITICAL: attach listener FIRST
                this.port.onmessage = (e) => this.handleCommand(e.data);

                this.port.start();
                this.isShared = true;

                // NOW send init — guaranteed after connection
                this.port.postMessage({
                    type: 'init',
                    options: this.options
                });
                console.log('[SessionKeeper] SharedWorker loaded');
            } catch (err) {
                console.warn('[SessionKeeper] SharedWorker failed, falling back:', err);
                this.setupDedicatedWithBroadcast(workerUrl);
            }
        } else {
            console.log('[SessionKeeper] No SharedWorker support, using dedicated + BroadcastChannel');
            this.setupDedicatedWithBroadcast(workerUrl);
        }
        console.log('%c[SessionKeeper] Using ' + (this.isShared ? 'SharedWorker' : 'Dedicated Worker + BroadcastChannel'), 'color: #2196F3; font-weight: bold;');

        this.startHeartbeat();
    }

    startHeartbeat() {
        if (this.heartbeatTimer) {
            clearInterval(this.heartbeatTimer);
        }
        const heartbeatInterval = 30000;  // Every 30s
        let lastPong = Date.now();

        // Save the current handler
        const currentHandler = this.port.onmessage;

        // Override to catch pong while preserving everything else
        this.port.onmessage = (e) => {
            // Always call the existing handler first (handleCommand or previous override)
            if (currentHandler) {
                currentHandler.call(this.port, e);
            }

            // Then check for pong
            if (e.data && e.data.type === 'pong') {
                lastPong = Date.now();
            }
        };

        const heartbeat = () => {
            if (Date.now() - lastPong > 60000) {  // No pong in 60s -> dead
                console.warn('[SessionKeeper] Worker unresponsive - restarting');
                this.initWorker();  // Re-initialize (creates new worker/port)
                return;
            }
            this.port.postMessage({ type: 'ping' });
        };

        this.heartbeatTimer = setInterval(heartbeat, heartbeatInterval);
        heartbeat();  // Initial ping
        window.addEventListener('unload', () => clearInterval(this.heartbeatTimer));
    }

    setupDedicatedWithBroadcast(workerUrl) {
        this.worker = new Worker(workerUrl, { name: 'SessionKeeperDedicatedWorker' });
        this.port = this.worker;
        this.isShared = false;

        // Setup BroadcastChannel for cross-tab sync
        this.broadcast = new BroadcastChannel('sessionkeeper_sync');
        this.broadcast.onmessage = (e) => {
            if (e.data === 'activity' || e.data === 'renew_success') {
                // Forward foreign tab events to our local worker
                this.port.postMessage({ type: e.data === 'activity' ? 'activity' : 'renew' });
            }
        };
    }

    handleCommand(cmd) {
        switch (cmd.type) {
            case 'showWarning':
                this.uiHandler.showWarning?.();
                break;
            case 'showExpired':
                this.uiHandler.showExpired?.();
                break;
            case 'closeWarning':
                this.uiHandler.closeWarning?.();
                break;
            case 'countdown':
                window.dispatchEvent(new CustomEvent('SessionKeeperCountdown', {
                    detail: cmd.time // {h,m,s}
                }));
                break;
            case 'redirect':
                setTimeout(() => {
                    if (this.options.redirect) {
                        location.href = this.options.redirect;
                    } else {
                        location.reload();
                    }
                }, 60000);  // do whatever after 60s, gives the session time to expire
                break;
            case 'status':
                if (this.options.debug) {
                    console.log('%c[SessionKeeper Worker] ' + cmd.message, 'color: #4CAF50; font-weight: bold;');
                }
                break;
        }
    }

    // Exposed for UI buttons (e.g., "Stay logged in")
    renewSession() {
        this.port.postMessage({ type: 'renew' });

        // In dedicated mode, broadcast so other tabs reset too (worker will confirm success)
        if (!this.isShared && this.broadcast) {
            this.broadcast.postMessage('activity'); // or a specific 'renew' if you prefer
        }
    }
}

// Restore legacy XHR detection (previously in hijackxhr.js)
(function() {
    const originalSend = XMLHttpRequest.prototype.send;

    XMLHttpRequest.prototype.send = function() {
        const xhr = this;
        const oldCallback = xhr.onreadystatechange;

        xhr.onreadystatechange = function() {
            if (xhr.readyState === 4 && xhr.responseURL) {
                const evt = new CustomEvent('SessionKeeperXHROpen', { detail: xhr.responseURL });
                window.dispatchEvent(evt);
            }

            if (oldCallback) {
                oldCallback.apply(xhr, arguments);
            }
        };

        return originalSend.apply(xhr, arguments);
    };
})();

// Global instance so UI scripts can reach it if needed
window.addEventListener('DOMContentLoaded', function() {
    if (!window.SessionKeeper) {
        new SessionKeeperClient();
        window.SessionKeeper.reportActivity(document.location.href);
        // Global fetch hook to catch Joomla keepalive (and any other fetch)
        (function() {
            if (!window.fetch) return; // Safety

            const originalFetch = window.fetch;
            window.fetch = function(resource, init) {
                let url = typeof resource === 'string' ? resource : resource.url || '';

                // Optional: log for debug
                // console.log('[SessionKeeper Debug] Fetch detected:', url);

                // Trigger activity on keepalive-specific URLs
                if (url.includes('option=com_ajax') || url.includes('format=json')) {
                    if (window.SessionKeeper?.port) {
                        window.dispatchEvent(new CustomEvent('SessionKeeperXHROpen', {
                            detail: url
                        }));
                    }
                }

                // Always call original
                return originalFetch.apply(this, arguments);
            };
            console.log('[SessionKeeper] Native fetch hooked for keepalive detection');
        })();
    }
});
