class SessionKeeperCore {
    constructor() {
        this.options = null;
        this.warningTimeout = null;
        this.expiredTimeout = null;
        this.countdownInterval = null;
        this.countdownTarget = null;
        this.ports = new Set(); // Only used in SharedWorker mode
    }

    // ---------- Utils ----------
    s2hms(seconds) {
        if (seconds < 0) seconds = 0;
        const h = Math.floor(seconds / 3600);
        seconds %= 3600;
        const m = Math.floor(seconds / 60);
        const s = Math.floor(seconds % 60);
        return { h, m, s };
    }

    zeroPad(num, places = 2) {
        return String(num).padStart(places, '0');
    }

    getRemainingWarningSeconds() {
        if (!this.countdownTarget) return 0;
        const diff = (this.countdownTarget.getTime() - Date.now()) / 1000;
        return Math.max(0, diff);
    }

    // ---------- Core Logic ----------
    init(options) {
        this.options = options;
        this.status('Worker initialized with timeout: ' + options.timeout + ' min, warning: ' + options.warning + ' min');
        this.resetTimers();
    }

    resetTimers() {
        clearTimeout(this.warningTimeout);
        clearTimeout(this.expiredTimeout);
        clearInterval(this.countdownInterval);

        this.status('Timers reset - next warning in ' + (this.options.timeout - this.options.warning) + ' min');

        const warnDelay = (this.options.timeout - this.options.warning) * 60000;
        const expireDelay = this.options.timeout * 60000;

        this.warningTimeout = setTimeout(() => this.triggerWarning(), warnDelay);
        this.expiredTimeout = setTimeout(() => this.triggerExpired(), expireDelay);
    }

    triggerWarning() {
        this.startCountdown();
        this.status('Warning triggered - showing UI');
        this.broadcast({ type: 'showWarning' });
    }

    triggerExpired() {
        clearInterval(this.countdownInterval);
        this.status('Session expired - showing expired UI');
        this.broadcast({ type: 'showExpired' });
        // Optional auto-redirect after delay
        setTimeout(() => this.broadcast({ type: 'redirect' }), 5000);
    }

    startCountdown() {
        const now = Date.now() + this.options.warning * 60000;
        this.countdownTarget = new Date(now);

        this.countdownInterval = setInterval(() => {
            const secs = this.getRemainingWarningSeconds();
            const time = this.s2hms(secs);
            this.broadcast({
                type: 'countdown',
                time,
                formatted: `${this.zeroPad(time.h)}:${this.zeroPad(time.m)}:${this.zeroPad(time.s)}`
            });

            if (secs <= 0) {
                clearInterval(this.countdownInterval);
            }
        }, 1000);
    }

    // renewSession() {
    //     this.status('Attempting session renewal...');
    //     fetch(this.options.renewurl, {
    //         method: 'GET',
    //         credentials: 'same-origin',
    //         headers: {
    //             'X-Requested-With': 'XMLHttpRequest',
    //             'Accept': 'application/json'
    //         }
    //     })
    //     .then(r => r.json())
    //     .then(() => {
    //         this.status('Session successfully renewed');
    //         this.resetTimers();
    //         this.broadcast({ type: 'closeWarning' });
    //     })
    //     .catch(err => {
    //         this.status('Session renew failed: ' + err.message);
    //         console.error('Renew failed', err);
    //     });
    // }
    async renewSession() {
        this.status('Attempting session renewal...');

        const maxRetries = 4;  // Includes the initial attempt
        let attempt = 0;

        while (attempt < maxRetries) {
            try {
                const response = await fetch(this.options.renewurl, {
                    method: 'GET',
                    credentials: 'same-origin',
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest',
                        'Accept': 'application/json'
                    }
                });

                if (!response.ok) throw new Error(`HTTP ${response.status}`);

                await response.json();  // Assuming it returns something valid

                this.status('Session successfully renewed');
                this.resetTimers();
                this.broadcast({ type: 'closeWarning' });
                return;  // Success!
            } catch (err) {
                attempt++;
                this.status(`Renew attempt ${attempt} failed: ${err.message}`);

                if (attempt >= maxRetries) {
                    this.status('All renew attempts failed');
                    // Optional: broadcast a 'renewFailed' command for UI to show error
                    this.broadcast({ type: 'renewFailed' });
                    return;
                }

                // Exponential backoff with jitter (1s * 2^(attempt-1) + random)
                const delay = 1000 * Math.pow(2, attempt - 1) + Math.random() * 1000;
                await new Promise(resolve => setTimeout(resolve, delay));
            }
        }
    }

    // ---------- Messaging ----------
    broadcast(message) {
        if (this.ports.size > 0) {
            // SharedWorker mode
            this.ports.forEach(port => port.postMessage(message));
        } else {
            // Dedicated mode
            postMessage(message);
        }
    }

    status(msg) {
        this.broadcast({ type: 'status', message: msg });
    }

    handleMessage(msg, port) {
        switch (msg.type) {
            case 'init':
                this.init(msg.options);
                break;
            case 'activity':
                this.status('Activity detected - resetting timers: '+ msg.url);
                this.resetTimers();
                this.broadcast({ type: 'closeWarning' });
                break;
            case 'renew':
                this.renewSession();
                break;
            case 'ping':
                this.broadcast({ type: 'pong' });
                break;
        }
    }
}

// ---------- Worker Entry Point ----------
const core = new SessionKeeperCore();

// Add an initial status for debugging – this will fire in both modes safely
// (We'll handle broadcast properly below)

// Reliable way: always set up onconnect if possible, and fallback for dedicated
self.onconnect = (e) => {
    console.log('[SessionKeeper Worker] onconnect fired!'); // Early log to confirm trigger

    const port = e.ports[0];
    core.ports.add(port);
    port.start();

    // Send per-port status
    port.postMessage({ type: 'status', message: 'SharedWorker connected - port open' });

    port.onmessage = (msg) => core.handleMessage(msg.data, port);

    port.onclose = () => core.ports.delete(port);
};

// Fallback for Dedicated Worker mode (runs if no connections ever come in)
if (typeof SharedWorkerGlobalScope === 'undefined' || !(self instanceof SharedWorkerGlobalScope)) {
    // Dedicated mode – use global onmessage/postMessage
    self.postMessage({ type: 'status', message: 'Dedicated Worker started' });

    onmessage = (e) => core.handleMessage(e.data);
} else {
    // Confirmed SharedWorker – optional extra status if you want
    // (No top-level postMessage needed – ports will handle it)
}