<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  System.sessionkeeper
 *
 * @copyright   (C) 2020 Michael Richey. <https://www.richeyweb.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\Sessionkeeper\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Event\SubscriberInterface;
use Joomla\Event\DispatcherInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Session\Session;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * System plugin to add additional security features to the administrator interface.
 *
 * @since  5.0.0
 */
final class Sessionkeeper extends CMSPlugin
{
    protected $app;
    protected $doc;
    protected $wa;
	private $_body = false;
    private $_debug = false;

    /**
     * Load the language file on instantiation.
     *
     * @var    boolean
     * @since  3.1
     */
    protected $autoloadLanguage = true;

    /**
     * Returns an array of events this subscriber will listen to.
     *
     * @return  array
     *
     * @since   4.0.0
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'onBeforeCompileHead' => 'onBeforeCompileHead',
            'onAfterRender' => 'onAfterRender'
        ];
    }

    public function __construct(
        DispatcherInterface $dispatcher,
        array $config
    ) {
        parent::__construct($dispatcher, $config);
        $this->app = Factory::getApplication();
		$this->doc = Factory::getDocument();
        $this->wa = $this->doc->getWebAssetManager();
        $this->_debug = $this->app->get('debug');
    }

	function onBeforeCompileHead() {
		$user = $this->app->getIdentity();

		if (!$user->id || $this->doc->getType() != 'html') {
			return;
		}

		if(count(array_intersect($user->groups,$this->params->get('usergroups',array())))) {
		    HTMLHelper::_('behavior.keepalive');
		    return;
		}

		if($this->params->get('showwarning',1)) {
			$this->_setWarning();
		}
    }

	function onAfterRender() {
		if($this->_body === false) {
			return;
		}
		$body = $this->app->getBody();
		$this->app->setBody(str_replace('</body>',$this->_body.'</body>',$body));
    }

	private function _getRedirect() {
		$menu = $this->app->getMenu();
		$redirect = $this->app->isClient('administrator') ? false : $this->params->get('redirectitemid', false);
		if ((bool)$redirect)
		{
			$item = $menu->getItem($redirect);
			return Route::_('index.php?' . $item->link . '&Itemid=' . $redirect, false);
		}
		return false;
	}

	private function _setWarning() {
        // $globalconfig = $this->app->getConfig();]
        $config = ['timeout' => (int) $this->app->get('lifetime')];
        $config['debug'] = $this->_debug;
		$config['warning'] = (int) $this->params->get('advancewarning', 2);
		$config['renewurl'] = Route::_('index.php?option=com_ajax&format=json',false);
		$config['redirect'] = $this->_getRedirect();
        $this->wa->registerAndUseScript('plg_system_sessionkeeper', 'media/plg_system_sessionkeeper/js/sessionkeeper.js', ['version' => 'auto']);
		$type = $this->params->get('messagetype', 'confirm');
		$this->{'_setup' . ucfirst($type)}($config);
	}

    private function _setupDialog($config) {
		$this->_body = $this->_loadTemplate('dialog', $config);
        $this->wa->registerAndUseScript('plg_system_sessionkeeper_dialog', 'media/plg_system_sessionkeeper/js/dialog.js', ['version' => 'auto']);
		$this->doc->addScriptOptions('plg_system_sessionkeeper', $config);
    }

    private function _setupConfirm($config) {
        foreach(['CONFIRMWARNINGMESSAGE','EXPIREDMESSAGE'] as $key) {
            $this->_scriptText($key);
        }
        $this->wa->registerAndUseScript('plg_system_sessionkeeper_confirm', 'media/plg_system_sessionkeeper/js/confirm.js', ['version' => 'auto']);
		$this->doc->addScriptOptions('plg_system_sessionkeeper', $config);
    }

    private function _setupEvent($config) {
        $this->wa->registerAndUseScript('plg_system_sessionkeeper_event', 'media/plg_system_sessionkeeper/js/event.js', ['version' => 'auto']);
		$this->doc->addScriptOptions('plg_system_sessionkeeper', $config);
    }

    private function _setupAlert($config) {
		Text::script('JCANCEL');
        foreach(['WARNINGMESSAGE','EXPIREDMESSAGE','CLOSE','RESCUE'] as $key) {
            $this->_scriptText($key);
        }
        $this->wa->registerAndUseScript('plg_system_sessionkeeper_alert', 'media/plg_system_sessionkeeper/js/alert.js', ['version' => 'auto']);
		$this->doc->addScriptOptions('plg_system_sessionkeeper', $config);
    }

    private function _setupToast($config) {
        switch($this->params->get('toastposition','top-end')) {
            case 'top-start':
                $config['position'] = 'position-fixed top-0 start-0';
                break;
            case 'top-center':
                $config['position'] = 'position-fixed top-0 start-50 translate-middle-x';
                break;
            case 'top-end':
                $config['position'] = 'position-fixed top-0 end-0';
                break;
            case 'center-start':
                $config['position'] = 'position-fixed top-50 start-0 translate-middle-y';
                break;
            case 'center-center':
                $config['position'] = 'position-fixed top-50 start-50 translate-middle';
                break;
            case 'center-end':
                $config['position'] = 'position-fixed top-50 end-0 translate-middle-y';
                break;
            case 'bottom-start':
                $config['position'] = 'position-fixed bottom-0 start-0';
                break;
            case 'bottom-center':
                $config['position'] = 'position-fixed bottom-0 start-50 translate-middle-x';
                break;
            case 'bottom-end':
                $config['position'] = 'position-fixed bottom-0 end-0';
                break;
        }
		$this->_body = $this->_loadTemplate('toast', $config);
		HTMLHelper::_('bootstrap.toast', '.sessionkeeper-toast', []);
        foreach(['WARNINGMESSAGE','EXPIREDMESSAGE','ABANDON','RESCUE','MODALTITLE'] as $key) {
            $this->_scriptText($key);
        }
        // $this->doc->addScript(Uri::root(true).'/media/plg_system_sessionkeeper/js/toast'.($this->_debug?'':'.min').'.js', ['version' => 'auto']);
        $this->wa->registerAndUseScript('plg_system_sessionkeeper_toast', 'media/plg_system_sessionkeeper/js/toast.js', ['version' => 'auto']);
		$this->doc->addScriptOptions('plg_system_sessionkeeper', $config);
    }

	private function _setupModal($config) {
		$this->_body = $this->_loadTemplate('modal', $config);
		HTMLHelper::_('bootstrap.modal', '.selector', []);
        // $this->doc->addScript(Uri::root(true).'/media/plg_system_sessionkeeper/js/modal'.($this->_debug?'':'.min').'.js', ['version' => 'auto']);
        $this->wa->registerAndUseScript('plg_system_sessionkeeper_modal', 'media/plg_system_sessionkeeper/js/modal.js', ['version' => 'auto']);
		$this->doc->addScriptOptions('plg_system_sessionkeeper', $config);
	}

	private function _loadTemplate($name, $config) {
        $path = PluginHelper::getLayoutPath('system', 'sessionkeeper', $name);
        ob_start();
        include $path;
        $html = ob_get_clean();
        return $html;
	}

    private function _scriptText($key) {
        return Text::script('PLG_SYSTEM_SESSIONKEEPER_' . strtoupper($key), true, true);
    }


}
