<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  System.xautopost
 *
 * @copyright   (C) 2025 Michael Richey. <https://www.richeyweb.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace RicheyWeb\Plugin\System\XAutopost\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Event\Event;
use Joomla\CMS\Event\Content\ContentPrepareEvent;
use Joomla\CMS\Event\Model\PrepareFormEvent;
use Joomla\CMS\Event\Model\PrepareDataEvent;
use Joomla\CMS\Event\Model\AfterSaveEvent;
// use Joomla\CMS\Event\Model\BeforeSaveEvent;
use Joomla\CMS\Event\Model\AfterDeleteEvent;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Registry\Registry;
use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\Event\SubscriberInterface;
use Joomla\CMS\Event\Result\ResultAwareInterface;
use Joomla\Event\DispatcherInterface;
use Joomla\CMS\Application\SiteApplication;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Router\Router;


// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Joomla! System Logging Plugin.
 *
 * @since  1.5
 */
final class XAutopost extends CMSPlugin
{
    private $_cache = false;
    private $caching = true;
    protected $app;
    protected $db;
    protected $autoloadLanguage = true;

    /**
     * The supported form contexts
     *
     * @var    array
     *
     * @since  3.9.0
     */
    protected $supportedContext = [
        'com_content.article',
        'com_contact.contact',
    ];

    protected $fields = [
        'show_title' => 'INTEGER',
        'show_description' => 'INTEGER',
        'custom_tweet' => 'STRING',
        'card_type' => 'STRING',
        'site_option' => 'STRING',
        'site' => 'CMD',
        'site_id_option' => 'STRING',
        'site_id' => 'INTEGER',
        'creator_option' => 'STRING',
        'creator' => 'CMD',
        'description_option' => 'STRING',
        'description' => 'STRING',
        'title_option' => 'STRING',
        'title' => 'STRING',
        'image_option' => 'STRING',
        'image' => 'STRING',
        'image_alt_option' => 'STRING',
        'image_alt' => 'STRING',
        'player' => 'STRING',
        'player_stream' => 'STRING',
        'player_width' => 'INTEGER',
        'player_height' => 'INTEGER',
        'app_country' => 'STRING',
        'app_name_iphone' => 'STRING',
        'app_id_iphone' => 'STRING',
        'app_url_iphone' => 'STRING',
        'app_name_ipad' => 'STRING',
        'app_id_ipad' => 'STRING',
        'app_url_ipad' => 'STRING',
        'app_name_googleplay' => 'STRING',
        'app_id_googleplay' => 'STRING',
        'app_url_googleplay' => 'STRING'        
    ];

    /**
     * Constructor
     *
     * @param   DispatcherInterface              $dispatcher                 The object to observe
     * @param   array   $config    An array that holds the plugin configuration
     *
     * @since   1.5
     */
    public function __construct(
        DispatcherInterface $dispatcher,
        array $config
    )
    {
        parent::__construct($dispatcher, $config);    
        if (!$this->app instanceof SiteApplication) {
            return false;
        }
        // setup cache
        if(!$this->_cache) {
            $options = [
                'defaultgroup' => 'plg_system_xautopost',
                'caching' => true,
                'lifetime' => $this->app->get('cachetime', 15),
            ];
            $this->caching = (bool)$this->params->get('caching', 1);
            $this->_cache = Factory::getContainer()->get(CacheControllerFactoryInterface::class)
                ->createCacheController('output',$options);
        }
    }


    /**
     * Returns an array of events this subscriber will listen to.
     *
     * @return array
     *
     * @since   5.3.0
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'onAjaxXautopost' => 'onAjaxXautopost',
            'onBeforeCompileHead' => 'onBeforeCompileHead',
            'onContentAfterSave' => 'onContentAfterSave',
            // 'onContentBeforeSave' => 'onContentBeforeSave',
            'onContentPrepareForm' => 'onContentPrepareForm',
            'onContentPrepareData' => 'onContentPrepareData',
            'onContentPrepare' => 'onContentPrepare',
            'onContactAfterSave' => 'onContentAfterSave',
            'onContactBeforeSave' => 'onContentBeforeSave',
            'onContentAfterDelete' => 'onContentAfterDelete',
            'onContactAfterDelete' => 'onContentAfterDelete',
        ];
    }

    public function onAjaxXautopost(Event $event)
    {
        $input = $this->app->input->get('id',false,'INTEGER');
        $card = $this->retrieveTwitterCardById((int)$input);
        $data = $this->buildPlayer($card) ?? Text::_('PLG_SYSTEM_XAUTOPOST_ERROR_NO_PLAYER');
        // $event->addResult($data);
        die($data); // we save a tiny bit of processing by just dying here
    }

    public function onBeforeCompileHead()
    {
        $app = Factory::getApplication();
        if(!$app->isClient('site')) {
            return;
        }
        $doc = $app->getDocument();
        if($doc->getType() !== 'html') {
            return;
        }
        if(isset($doc->_metaTags['name']['twitter:card'])) {
            return;
        }

        $cachekey = 'generic_' . md5(Uri::current());

        if($this->caching && $this->_cache->contains($cachekey)) {
            $card = $this->_cache->get($cachekey);
        } else {
            $defaults = $this->params->get('defaults', (new \stdClass()));
            $cardtype = $defaults->card_type ?? 'summary';
            $show_title = $defaults->show_title ?? 1;
            $twittercard = [
                'show_title'=>$show_title,
                'card_type'=>$cardtype,
                'site_option'=>'default',
                'site_id_option'=>'default',
                'creator_option'=>'default',
                'creator_id_option'=>'default',
                'title_option'=>'custom',
                'title'=>$doc->title,
                'description_option'=>'custom',
                'description'=>$doc->description
            ];
            $cardValues = new \RicheyWeb\Plugin\System\XAutopost\XMetas\XMeta_generic((object)$twittercard,null,$this->params->get('defaults'));
            $card = $cardValues->getMetaTags();
            if(!$card){
                return;
            }
            if($this->caching && $this->_cache) {
                $this->_cache->store($card,$cachekey);
            }

        }

        if(empty($card) || !is_array($card)) {
            $this->_cache->clear($cachekey);
            return;
        }
        
        foreach($card as $key=>$meta) {
            if(!is_string($meta['name']) || !is_string($meta['content'])) {
                continue;
            }
            switch($key){
                case 'title':
                    $value = $this->trimTextWithEllipsis($meta['content'],70);
                    break;
                case 'description':
                    $value = $this->trimTextWithEllipsis($meta['content'],200);
                    break;
                case 'image_alt':
                    $value = $this->trimTextWithEllipsis($meta['content'],420);
                    break;
                default:
                    $value = $meta['content'];
                    break;
            }
            $doc->setMetaData($meta['name'], $value);
        }
    }

    // public function onContentBeforeSave(BeforeSaveEvent $event)
    // {
    //     $context = $event->getContext();
    //     $article = $event->getItem();

    //     // Check if the context is an article and it's a new article
    //     if (!in_array($context, $this->supportedContext) || !$article->id) {
    //         return true;
    //     }

    //     if($this->caching && $this->_cache){
    //         $this->_cache->clear($context.'_'.$article->id);
    //     }

    //     return true;
    // }

    public function onContentAfterSave(AfterSaveEvent $event)
    {
        $context = $event->getContext();
        $article = $event->getItem();
        $isNew = $event->getIsNew();


        // Check if the context is an article and it's a new article
        if (!in_array($context, $this->supportedContext)) {
            return true;
        }

        if($this->caching && $this->_cache){
            $this->_cache->remove($context.'_'.$article->id);
        }
        
        $data = $event->getData();

        $values = $this->storeTwitterCard($context,$data, $article->id);

        // Check if the article is published
        $property = $context === 'com_contact.contact' ? 'published' : 'state';
        if($article->{$property} != 1) {
            return true;
        }
        
        // check coverage
        // $coverage = $this->params->get('coverage', false);
        $coverage = $this->coverage($article, $context);
        if(!$values['submitted'] && $coverage) {
            $article->xautopost = (object)$data['xautopost'];
            $rv = $this->xAutopost($article,$context);
            if(is_string($rv)) {
                // error message
                $this->app->enqueueMessage(Text::sprintf('PLG_SYSTEM_XAUTOPOST_ERROR',$rv),'error');
                // break;
            } else {
                switch($rv){
                    case false:
                        // already posted or missing credentials, do nothing
                        break;
                    case true:
                        // $this->setCardSubmitted($data['xautopost']['xautopost_id']);
                        $this->setCardSubmitted($values['xautopost_id']);
                        $this->app->enqueueMessage(Text::_('PLG_SYSTEM_XAUTOPOST_SUCCESS'),'info');
                        break;
                }
            }
        }
        
        return true;
    }

    public function onContentAfterDelete(AfterDeleteEvent $event)
    {
        $context = $event->getContext();
        $article = $event->getItem();

        if (!in_array($context, $this->supportedContext) || !isset($article->id)) {
            return true;
        }

        $query = $this->db->createQuery(true)
            ->delete($this->db->quoteName('#__plg_system_xautopost'))
            ->where($this->db->quoteName('context') . ' = ' . $this->db->quote($context))
            ->where($this->db->quoteName('content_id') . ' = ' . (int) $article->id);
        
        $this->db->setQuery($query);
        $this->db->execute();

        return true;
    }

    private function buildPlayer($card){
        if(!$card) {
            return false;
        }
        if($this->caching && $this->_cache->contains('player_'.$card->id)) {
            return $this->_cache->get('player_'.$card->id);
        }
        $stream = $card->player_stream ?? '';
        if(empty($stream)) {
            return false;
        }
        $filename = basename($stream);
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        $mimetype = '';
        $rooturl = Uri::root(false);
        $background = false;
        switch($extension){
            case 'mp4':
            case 'm4v':
                $template = 'video';
                $mimetype = 'video/mp4';
                $translatedMessage = Text::_('PLG_CONTENT_XAUTOPLAY_VIDEO_FALLBACK');
                break;
            case 'mp3':
                $template = 'audio';
                $mimetype = 'audio/mpeg';
                $translatedMessage = Text::_('PLG_CONTENT_XAUTOPLAY_AUDIO_FALLBACK');
                $background = $rooturl.ltrim($card->image ?? '','/');
                break;
            case 'm4a':
            case 'm4b':
                $template = 'audio';
                $mimetype = 'audio/aac';
                $translatedMessage = Text::_('PLG_CONTENT_XAUTOPLAY_AUDIO_FALLBACK');
                $background = $rooturl.ltrim($card->image ?? '','/');
                break;
            }
        if(empty($mimetype)) {
            // unsupported mimetype
            return false;
        }
        $url = filter_var($stream, FILTER_VALIDATE_URL) ? $stream : $rooturl.'/'.ltrim($stream,'/');
        $path = PluginHelper::getLayoutPath('system', 'xautopost', $template);
        ob_start();
        include $path;
        $player = ob_get_clean();
        if($this->caching) {
            $this->_cache->store($player, 'player_'.$card->id);
        }
        return $player;
    }

    private function coverage($item, $context) {
        $coverage = $this->params->get('coverage', false);
        if(!$coverage) {
            return true;
        }
        $component = explode('.',$context)[0];
        foreach((array)$coverage as $cov) {
            if($cov->component !== $component) {
                // end this loop if the component doesn't match
                continue;
            }

            $property = $component.'_category';
            $selectedCategories = (array)($cov->{$property}??[]);
            // check if we're excluding categories and that there are categories to exclude
            if($cov->exclude_categories ?? 0 && count($selectedCategories)) {
                // exclude selected categories
                if(in_array($item->catid, $selectedCategories)) {
                    // cat id is in the excluded list
                    return false;
                }
                // cat id is not in the excluded list
                return true;
            }
            // if no categories selected, all categories are covered
            // if categories selected, only those categories are covered
            if(count($selectedCategories) && !in_array($item->catid, $selectedCategories)) {
                return false;
            }
            return true;
        }
        return false;
    }

    public function onContentPrepare(ContentPrepareEvent $event) {
        $context = $event->getContext();
        if (!in_array($context, $this->supportedContext)) {
            return true;
        }
        $view = $this->app->input->get('view',false);
        // we must be in a view that matches the context
        if ($view !== explode('.',$context)[1]) {
            return true;
        }
        $article = $event->getItem();

        if(!($article->id ?? false)){
            return true;
        }
        $cardClass = 'RicheyWeb\\Plugin\\System\\XAutopost\\XMetas\\XMeta_'.explode('.',$context)[0];

        // check cache for existing twitter card data
        $cachekey = $context . '_' . $article->id;
        if($this->caching && $this->_cache->contains($cachekey)) {
            $card = $this->_cache->get($cachekey);
        } else {
            $twittercard = $this->retrieveTwitterCard($context,$article->id);
            $cardValues = new $cardClass($twittercard,$article,$this->params->get('defaults', (new \stdClass())));
            $card = $cardValues->getMetaTags();
            if($this->caching) {
                $this->_cache->store($card, $cachekey);
            }
        }

        $doc = $this->app->getDocument();
        if(!empty($card)) {
            foreach($card as $key=>$meta) {
                if(!is_string($meta['name'])) {
                    continue;
                }
                $doc->setMetaData($meta['name'], (string)$meta['content']);
            }
        }
        
    }

    public function onContentPrepareForm(PrepareFormEvent $event)
    {
        // Check we have a valid form.
        $form = $event->getForm();
        $name = $form->getName();
        if (!\in_array($name, $this->supportedContext)) {
            return;
        }

        // we must be in an edit view
        if ($this->app->input->get('layout',false) !== 'edit') {
            return;
        }

        // Add the twittercard fieldset to the form.
        if(file_exists(__DIR__ . '/../Forms/twittercard.'.explode('.',$name)[0].'.xml')) {
            $form->loadFile(__DIR__ . '/../Forms/twittercard.'.explode('.',$name)[0].'.xml', false);
        } else {
            $form->loadFile(__DIR__ . '/../Forms/twittercard.xml', false);
        }

        $defaults = $this->params->get('defaults', (new \stdClass()));
        $tweetDefaults = $this->params->get('tweet_defaults', 0);
        
        // manipulating a few fields to set their defaults to the plugin configured default values
        $form->setFieldAttribute('show_title', 'default', $tweetDefaults->show_title ?? 1, 'xautopost');
        $form->setFieldAttribute('show_description', 'default', $tweetDefaults->show_description ?? 1, 'xautopost');
        $form->setFieldAttribute('submitted', 'default', $defaults->submitted, 'xautopost');
        $form->setFieldAttribute('app_country', 'default', $defaults->app_country ?? 'US', 'xautopost');
        $form->setFieldAttribute('player_width', 'default', $defaults->player_width ?? 640, 'xautopost');
        $form->setFieldAttribute('player_height', 'default', $defaults->player_height ?? 360, 'xautopost');

        return true;
    }

    public function onContentPrepareData(PrepareDataEvent $event){
        $context = $event->getContext();
        $data = $event->getData();

         // Check we have a valid context and data.

        if (!\in_array($context, $this->supportedContext)){
            return;
        }
        if (!isset($data->id)) {
            return;
        }
        
        $twittercard = $this->retrieveTwitterCard($context,$data->id);
        
        $data->xautopost = [];

        foreach((array)$twittercard as $field => $value) {
            $data->xautopost[$field] = $value;
        }
        $defaults = $this->params->get('defaults', (new \stdClass()));
        $data->xautopost['submitted'] = $twittercard->submitted ?? (int)$this->params->get('submitted', 0);
        $data->xautopost['app_country'] = $twittercard->app_country ?? $defaults->app_country ?? 'US';
        $data->xautopost['player_width'] = $twittercard->player_width ?? $defaults->player_width ?? 640;
        $data->xautopost['player_height'] = $twittercard->player_height ?? $defaults->player_height ?? 360;

        $event->updateData($data);
        return true;
    }

    private function getArticleUrl($article, $context) {
        $contextArray = explode('.',$context);
        $url = 'index.php?option=' . $contextArray[0] . '&view=' . $contextArray[1] . '&catid=' . $article->catid . '&id=' . $article->id;
        // we don't know, or care if we're in the frontend or backend, using Router instead of Route
        // because we can specify the intended client
        $router = Router::getInstance('site');
        $url = $router->build($url)->toString();
        $url = Uri::root() . ltrim($url, '/');
        return $url;
    }

    private function getCard($article, $context) {
        $cardClass = 'RicheyWeb\\Plugin\\System\\XAutopost\\XMetas\\XMeta_'.explode('.',$context)[0];
        
        // $card = Card::buildCard(
        $cardValues = new $cardClass(
            $this->retrieveTwitterCard($context,$article->id),
            $article,
            $this->params->get('defaults', (new \stdClass()))
        );
        return $cardValues->getMetaTags();
    }

    private function autopostText($title,$description,$url) {
        $remaining = 280 - strlen($url??'');
        $ellipsis = (bool)$this->params->get('ellipsis', 1);
        $status = '';
        if(strlen($title??'')) {
            $title = $ellipsis ? $this->trimTextWithEllipsis($title,70,false) : substr($title, 0, 70);
            $remaining -= strlen($title??'') + 2; // +2 for newlines
            $status .= $title;
        }
        if($remaining > 0 && strlen($description??'')) {
            if(strlen($status??'')) {
                $status .= "\n\n"; // add spacing if we already have a title
            }
            $remaining --; // for newline
            $description = $ellipsis ? $this->trimTextWithEllipsis($description, $remaining) : substr($description, 0, $remaining);        
            $status .= $description;
        }
        if(strlen($status??'')) {
            $status .= "\n"; // add spacing if we already have a title or description
        }
        // always add url
        $status .= $url;
        
        return $status;
    }
    private function trimTextWithEllipsis($text, $maxLength, $endings = ['.', '!', '?']) {
        // Input validation
        if (!is_string($text) || $maxLength < 1) {
            return '';
        }

        // Trim whitespace
        $text = trim($text);
        $textLength = strlen($text??'');

        // If text is empty, return empty
        if ($textLength === 0) {
            return '';
        }

        // Helper function to check if text ends with sentence-ending punctuation
        // (last non-whitespace char is ., !, or ?)
        $endsWithSentenceEnd = function($str,$endings) {
            if(!$endings) return true;
            $str = rtrim($str); // Remove trailing whitespace for accurate check
            if (strlen($str??'') === 0) return false;
            $lastChar = substr($str, -1);
            return in_array($lastChar, $endings);
        };

        // Check if text is already short enough and ends with sentence-ending punctuation
        if ($textLength <= $maxLength && $endsWithSentenceEnd($text, $endings)) {
            return $text; // Complete sentence, no ellipsis needed
        }

        // At this point, we need to add ellipsis (either because it's too long or doesn't end properly)
        // Account for ellipsis length (3 characters: ...)
        $effectiveMaxLength = $maxLength - 3;
        if ($effectiveMaxLength < 1) {
            return substr($text, 0, $maxLength); // Can't fit ellipsis, just truncate to max
        }

        // If text is already short enough, use full text as starting point for trimming (to make room if needed)
        // Otherwise, trim to effective max length first
        $trimmedText = ($textLength <= $maxLength) ? $text : substr($text, 0, $effectiveMaxLength);

        // Find last space within the trimmed text (to avoid word truncation)
        $lastSpace = 0;
        $pos = 0;
        $trimmedLength = strlen($trimmedText??'');
        while (($pos = strpos($trimmedText, ' ', $pos)) !== false && $pos < $trimmedLength) {
            $lastSpace = $pos;
            $pos++;
        }

        // Trim to last space (or full trimmed text if no spaces)
        $result = ($lastSpace > 0) ? substr($trimmedText, 0, $lastSpace) : $trimmedText;

        // Add ellipsis and ensure final length doesn't exceed maxLength
        $resultWithEllipsis = $result . '...';
        if (strlen($resultWithEllipsis??'') > $maxLength) {
            // Rare edge case: If still too long after word-boundary trim, hard-truncate to max
            return substr($resultWithEllipsis, 0, $maxLength);
        }

        return $resultWithEllipsis;
    }

    private function xAutopost($article, $context) {
        $url = $this->getArticleUrl($article, $context);
        $card = $this->getCard($article, $context);
        
        if(empty($card)) {
            return Text::_('PLG_SYSTEM_XAUTOPOST_ERROR_NO_CARD');
        }

        if(strlen(trim($article->xautopost->custom_tweet??'')) > 0) {
            $title = '';
            $description = trim($article->xautopost->custom_tweet);
        } else {
            $title = ($article->xautopost->show_title??$this->params->get('show_title', 1))?$card['title']['content']:'';
            $description = ($article->xautopost->show_description??$this->params->get('show_description', 1))?$card['description']['content']:'';
        }

        $status = $this->autopostText($title, $description, $url);
        if(empty($status)) {
            return Text::_('PLG_SYSTEM_XAUTOPOST_ERROR_NO_STATUS');
        }

        $result = $this->postToX($status);

        if (is_string($result)) {
            return $result; // return error message
        }

        return true;
    }

    private function postToX($status) {
        $consumer_key = $this->params->get('consumer_key', '');
        $consumer_secret = $this->params->get('consumer_secret', '');
        if (empty($consumer_key) || empty($consumer_secret)) {
            return Text::_('PLG_SYSTEM_XAUTOPOST_ERROR_NO_CONSUMER_CREDENTIALS');
        }

        $access_token = $this->params->get('access_token', '');
        $access_token_secret = $this->params->get('access_token_secret', '');
        if (empty($access_token) || empty($access_token_secret)) {
            return Text::_('PLG_SYSTEM_XAUTOPOST_ERROR_NO_ACCESS_CREDENTIALS');
        }

        // now that we have the credentials, proceed to post
        require_once(__DIR__ . '/vendor/autoload.php');

        $data = [
            'text' => $status
        ];

        $connection = new \Abraham\TwitterOAuth\TwitterOAuth(
            $consumer_key,
            $consumer_secret,
            $access_token,
            $access_token_secret
        );

        $connection->setApiVersion('2');

        try {
            $response = $connection->post('tweets', $data, ['jsonPayload' => true]);
            $httpStatus = $connection->getLastHttpCode();
            if ($httpStatus == 201) {
                return true; // success
            } else {
                if (isset($response->errors) && is_array($response->errors) && count($response->errors) > 0) {
                    $errorMessages = array_map(function($error) {
                        return isset($error->message) ? $error->message : 'Unknown error';
                    }, $response->errors);
                    return implode('; ', $errorMessages);
                }
                $httpStatus = $connection->getLastHttpCode();
                return Text::sprintf('PLG_SYSTEM_XAUTOPOST_ERROR_HTTP', $httpStatus, Text::_('PLG_SYSTEM_XAUTOPOST_ERROR_'.$httpStatus));
            }
        } catch (\Exception $e) {
            return Text::sprintf('PLG_SYSTEM_XAUTOPOST_ERROR_POSTING', $e->getMessage());
        }

        return true;
    }

    private function setCardSubmitted($xautopost_id) {
        $dbobject = new \stdClass();
        $dbobject->id = $xautopost_id;
        $dbobject->submitted = 1;

        $this->db->updateObject('#__plg_system_xautopost', $dbobject, 'id');
    }

    private function retrieveTwitterCard($context,$content_id) {
        $query = $this->db->createQuery(true)
            ->select($this->db->quoteName(['id', 'config','submitted']))
            ->from($this->db->quoteName('#__plg_system_xautopost'))
            ->where($this->db->quoteName('context') . ' = ' . $this->db->quote($context))
            ->where($this->db->quoteName('content_id') . ' = ' . (int) $content_id);
        
        $this->db->setQuery($query);
        $result = $this->db->loadObject();

        if ($result) {
            $config = json_decode($result->config, true);
            $config['submitted'] = $result->submitted;
            $config['xautopost_id'] = $result->id;
            return (object) $config;
        } else {
            $defaults = $this->params->get('defaults', (new \stdClass()));
            $defaultData = [
                'xautopost_id' => 0,
                'card_type' => 'default',
                'submitted' => (int)$this->params->get('submitted', 0),
                'app_country' => $defaults->app_country ?? 'US',
                'player_width' => $defaults->player_width ?? 640,
                'player_height' => $defaults->player_height ?? 360,
            ];
            // return (object) array_merge(['xautopost_id' => 0], ['card_type' => 'default', 'submitted' => $this->params->get('submitted', 0)] );
            return (object)$defaultData;
        }

    }

    private function retrieveTwitterCardById($xautopost_id) {
        $query = $this->db->createQuery(true)
            ->select($this->db->quoteName(['id', 'config','submitted']))
            ->from($this->db->quoteName('#__plg_system_xautopost'))
            ->where($this->db->quoteName('id') . ' = ' . (int) $xautopost_id);
        
        $this->db->setQuery($query);
        $result = $this->db->loadObject();

        if ($result) {
            $config = json_decode($result->config, true);
            $config['submitted'] = $result->submitted;
            return (object) array_merge(['xautopost_id' => $result->id], $config);
        } else {
            return false;
        }

    }

    private function storeTwitterCard($context,$data, $content_id) {
        $values = $this->fields;
        foreach ($values as $field => $type) {
            if (isset($data['xautopost'][$field])) {
                $values[$field] = $data['xautopost'][$field];
            }
        }
        
        $defaultSubmitted = $this->params->get('submitted', 0);
        
        $dbobject = new \stdClass();
        $dbobject->id = $data['xautopost']['xautopost_id'];
        $dbobject->context = $context;
        $dbobject->content_id = $content_id;
        $dbobject->config = json_encode($values);
        $dbobject->submitted = isset($data['xautopost']['submitted']) ? (int)$data['xautopost']['submitted'] : $defaultSubmitted;

        if($dbobject->id) {
            $this->db->updateObject('#__plg_system_xautopost', $dbobject, 'id');
        } else {
            $dbobject->id = 0;
            $this->db->insertObject('#__plg_system_xautopost', $dbobject, 'id');
        }
        $values['xautopost_id'] = $dbobject->id;
        $values['submitted'] = isset($data['xautopost']['submitted']) ? (int)$data['xautopost']['submitted'] : $defaultSubmitted;

        return $values;
    }


}