<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  User.staticemail
 *
 * @copyright   (C) 2020 Open Source Matters, Inc. <https://www.joomla.org>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\User\IndexingAPIUser\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Factory;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Event\AbstractEvent;
use Joomla\Component\Contact\Site\Helper\RouteHelper as ContactRouteHelper;


// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * An example custom terms and conditions plugin.
 *
 * @since  3.9.0
 */
final class IndexingAPIUser extends CMSPlugin
{
    protected $app;
    public function onUserAfterSave($user, $isnew, $success, $msg) {
        // best practices, exit early tests first
        if(!$user['id'] || !$success || $isnew){
            // if no user id, or the save failed, or the user is new, we don't want to process
            return true;
        }

        // now we setup $this->app and get a user object
        $this->app = Factory::getApplication();
        $juser = $this->app->getIdentity($user['id']);

        // test the user object to see if it's a valid user
        if(!$juser->id) {
            // user doesn't exist
            return true;
        }

        // get the user's groups
        $usergroups = $juser->groups;
        $validgroups = $this->params->get('validgroups', array());

        // test that the user is in a group we want to process
        if(!array_intersect($usergroups, $validgroups)) {
            // user isn't in a group we're monitoring
            return true;
        }

        // yay, we have a user we want to process

        // grab the contact records for this user
        $contacts = $this->getUserContactRecords($user['id']);
        if(!$contacts) {
            // user doesn't have a contact record
            return true;
        }

        foreach($contacts as $contact){
            // if the record isn't published or isn't guest/public, we don't want to index it
            if($contact->published == 0 || !in_array($contact->access,[1,5])) {
                // contact record is unpublished or not public
                return true;
            }
            // send the contact to the indexing API
            $this->sendToIndexingAPI($contact);
        }
    }

    private function getUserContactRecords($userid) {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('id,catid,language,published,access')
            ->from($db->quoteName('#__contact_details'))
            ->where($db->quoteName('user_id') . ' = ' . (int) $userid);
        $db->setQuery($query);
        return $db->loadObjectList()??false;
    }

    // send the contact to the indexing API
    private function sendToIndexingAPI($contact) {
        // indexing api external triggers need only an unrouted link property, that's all it needs because
        // ONLY plg_content_indexingapi is being dispatched, we can omit the normal content object properties
        // a content plugin normally expects
        $data = (object)[
            'link' => ContactRouteHelper::getContactRoute($contact->id, $contact->catid, $contact->language)
        ];
        $contentEventArguments = [
            'context' => 'IndexingAPIExternalTrigger',
            'subject' => $data,
            'isNew' => false  
        ];
        $dispatcher = $this->app->getDispatcher();
        PluginHelper::importPlugin('content','indexingapi',true, $dispatcher);
        $dispatcher->dispatch('onContentAfterSave', AbstractEvent::create('onContentAfterSave',$contentEventArguments));
    }
}