<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  User.logbadpasswords
 *
 * @copyright   (C) 2020 Michael Richey. <https://www.richeyweb.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace RicheyWeb\Plugin\User\LogBadPasswords\Extension;

use Joomla\CMS\Form\Form;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Factory;
use Joomla\Utilities\ArrayHelper;
use Joomla\CMS\Environment\Browser;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\User\UserFactoryInterface;
use Joomla\CMS\Form\FormHelper;
use Joomla\Database\DatabaseAwareTrait;



// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * An example custom terms and conditions plugin.
 *
 * @since  3.9.0
 */
final class LogBadPasswords extends CMSPlugin
{
    use DatabaseAwareTrait;

    protected $app;

    /**
     * Load the language file on instantiation.
     *
     * @var    boolean
     * @since  3.1
     */
    protected $autoloadLanguage = true;    

    public function onContentPrepareData($context, $data)
    {
        $app = Factory::getApplication();
        if ($app->isClient('site'))
            return true;

        // Check we are manipulating a valid form.
        if (!in_array($context, array('com_users.profile', 'com_users.user', 'com_users.registration', 'com_admin.profile'))) {
            return true;
        }
        if (is_object($data)) {
            $userId = isset($data->id) ? $data->id : 0;

            if (!isset($data->logbadpasswords) and $userId > 0) {
                // Load the profile data from the database.
                $db    = $this->getDatabase();
                $query = $db->createQuery(true);
                $query->select('profile_value')->from('#__user_profiles');
                $query->where('profile_key = ' . $db->quote('logbadpasswords.passwords') . ' AND user_id = ' . $userId);
                $db->setQuery($query);
                $badpasswords = $db->loadResult()??'';

                // Merge the profile data.

                FormHelper::addFieldPrefix('RicheyWeb\\Plugin\\User\\LogBadPasswords\\Field');
                FormHelper::addFormPath(JPATH_ROOT . '/plugins/user/logbadpasswords/profiles');
                $form = Form::getInstance('plg_user_logbadpasswords.form', 'logbadpasswords');

                $data->logbadpasswords = array('badpasswords' => $badpasswords);
            }
        }
        return true;
    }
    
    public function onContentPrepareForm($form, $data)
    {
        $app = Factory::getApplication();
        if ($app->isClient('site'))
            return true;

        if (!($form instanceof Form)) {
            $this->_subject->setError('JERROR_NOT_A_FORM');
            return false;
        }

        // Check we are manipulating a valid form.
        $name = $form->getName();
        if (!in_array($name, array('com_admin.profile', 'com_users.user', 'com_users.profile', 'com_users.registration'))) {
            return true;
        }

        // Add the registration fields to the form.
        FormHelper::addFieldPrefix('Joomla\\Plugin\\User\\LogBadPasswords\\Field');
        FormHelper::addFormPath(JPATH_ROOT . '/plugins/user/logbadpasswords/profiles');
        $form->loadFile('logbadpasswords', false);
    }

    public function onUserLogin($user, $options)
    {
        $app = Factory::getApplication();
        // some basic tests
        if ($app->isClient('administrator')) {
            if (!$this->params->get('runinadmin', false)) {
                return true;
            }
        }else {
            if (!$this->params->get('runinsite', false)) {
                return true;
            }
        }        

        // if there aren't any bad passwords to store, then we exit
        $newbadpasswords = $app->getUserState('plg_user_logbadpasswords', array());
        if (!count($newbadpasswords))
            return true;

        // now we need a db object
        $db    = $this->getDatabase();

        // we need the user id to continue
        $query = $db->createQuery(true);
        $query->select('id')->from('#__users')->where('username = ' . $db->quote($user['username']));
        $db->setQuery($query);
        $user['id'] = $db->loadResult();
        
        // now we verify that we're supposed to store the password for this user
        $juser = Factory::getUser($user['id']);
        foreach($juser->groups as $group) {
            if(in_array($group,$this->params->get('badgroups',array()))) return true;
        }

        // now we check to see if there were any previously stored bad passwords
        $query = $db->createQuery(true);
        $query->select(array('profile_value','user_id'))->from('#__user_profiles')->where('profile_key = "logbadpasswords.passwords" AND user_id = ' . $user['id']);
        $db->setQuery($query);
        $oldbadpasswords = $db->loadObject()??false;
        if ($oldbadpasswords) {
            $query = $db->createQuery(true);
            $oldbadpasswords = explode(', ', trim($oldbadpasswords->profile_value));
            foreach ($oldbadpasswords as $bp)
                if(strlen(trim($bp)) && !in_array(trim($bp),$newbadpasswords)) $newbadpasswords[] = trim($bp);
            $badpasswords = implode(', ', $this->_removeValid($user['password'],$newbadpasswords));
            $query->update('#__user_profiles')->set('profile_value = ' . $db->quote($badpasswords))->where('profile_key = "logbadpasswords.passwords" AND user_id = ' . $user['id']);
        } else {
            $query = $db->createQuery(true);
            $badpasswords = implode(', ', $this->_removeValid($user['password'],$newbadpasswords));
            $query->insert('#__user_profiles (user_id,profile_key,profile_value)')->values($user['id'] . ',' . $db->quote('logbadpasswords.passwords') . ',' . $db->quote($badpasswords));
        }
        $db->setQuery($query);
        $db->execute();
        return true;
    }
    public function onUserLoginFailure($credentials)
    {
        $app = Factory::getApplication();
        if($app->isClient('administrator') && $this->params->get('runinadmin', false, 'BOOLEAN')) {
            $this->_logBadPassword($credentials);
        }
        if($app->isClient('site') && $this->params->get('runinsite', false, 'BOOLEAN')) {
            $this->_logBadPassword($credentials);
        }
    }

    public function onUserAfterSave($user, $isnew, $success, $msg)
    {
        // you can only modify this from admin
        $app = Factory::getApplication();
        if ($app->isClient('site'))
            return true;

        $userId = ArrayHelper::getValue($user??[], 'id', 0, 'int');

        if ($userId && $success && isset($user['logbadpasswords']) && (count($user['logbadpasswords']))) {
            try {
                $db    = $this->getDatabase();
                $db->setQuery(
                        'DELETE FROM #__user_profiles WHERE user_id = ' . $userId .
                        " AND profile_key = 'logbadpasswords.passwords'"
                );

                if (!$db->execute()) {
                    throw new Exception($db->getErrorMsg());
                }

                $tuples = array();
                $order = 1;

                Form::addFormPath(dirname(__FILE__) . '/profiles');
                $form = Form::getInstance('plg_user_logpadpasswords.form', 'logbadpasswords');
                $values= '(' . $userId . ', ' . $db->quote('logbadpasswords.passwords') . ', ' . $db->quote($user['logbadpasswords']['badpasswords']) . ', ' . $order++ . ')';

                $db->setQuery('INSERT INTO #__user_profiles VALUES ' . $values);

                if (!$db->execute()) {
                    throw new Exception($db->getErrorMsg());
                }
            } catch (Exception $e) {
                $this->_subject->setError($e->getMessage());
                return false;
            }
        }

        return true;
    }

    public function onUserAfterDelete($user, $succes, $msg)
    {
        if (!$success) {
            return false;
        }

        $userId = ArrayHelper::getValue($user, 'id', 0, 'int');

        if ($userId) {
            try {
                $db    = $this->getDatabase();
                $db->setQuery(
                        'DELETE FROM #__user_profiles WHERE user_id = ' . $userId .
                        " AND profile_key = 'logbadpasswords.passwords'"
                );

                if (!$db->execute()) {
                    throw new Exception($db->getErrorMsg());
                }
            } catch (Exception $e) {
                $this->_subject->setError($e->getMessage());
                return false;
            }
        }

        return true;
    }

    private function _logBadPassword($credentials){
        $app = Factory::getApplication();
        $badpasswords = $app->getUserState('plg_system_logbadpasswords', array());
        if(!in_array($credentials['password'], $badpasswords)){
            $badpasswords[] = $credentials['password'];
        }
        $app->setUserState('plg_user_logbadpasswords', $badpasswords);
    }

    private function _removeValid($validpassword, $badpasswords) {
        if (in_array($validpassword, $badpasswords)) {
            unset($badpasswords[array_search($validpassword, $badpasswords)]);
        }
        return $badpasswords;
    }
}